/* VgaGames4 tutorial 7 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames4.h>
#include "pingpong.h"

/* object-collision functions */
extern void objcoll(void);

/* connect to nw-server */
extern int connect_to_nwserver(int, int);

static VG_BOOL show_help(void);


/* show help-text */
static VG_BOOL
show_help(void)
{
  char helptext[512];
  size_t hsize;
  struct VG_Image *imgp;
  int k_space;

  /* create text */
  snprintf(helptext, sizeof(helptext), "%%{txt&center[font=sysw:low fgcolor=0xffff00 bold=on]: Pingpong game%%}\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "\n\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [ALT+Q]%%}        Quit game\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [Cursor-UP]%%}    Move up\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [Cursor-DOWN]%%}  Move down\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [P]%%}            Pause\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [ESC]%%}          System-menu\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "\n\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt&center[fgcolor=0xdddddd]: Press space-key%%}\n");
  hsize = strlen(helptext);

  /* create image from text */
  imgp = vg4->font->totext(helptext, "[fgcolor=0xbbbb00 boxwidth=70%%]", NULL, NULL, NULL);
  if (imgp == NULL) { return VG_FALSE; }

  /* set space-key */
  if ((k_space = vg4->input->key_insert("Spacekey", VG_FALSE, VG_FALSE)) == 0) { return VG_FALSE; }
  vg4->input->key_setkbd(k_space, VG_INPUT_KBDCODE_SPACE);

  /* copy image to window */
  vg4->window->clear();
  vg4->window->copy(imgp, NULL, NULL);
  vg4->image->destroy(imgp);

  /* wait for pressing space-key */
  for (;;) {
    if (!vg4->input->update(VG_TRUE)) { return VG_FALSE; }
    if (vg4->input->key_newpressed(k_space)) { break; }
    vg4->window->flush();
    vg4->misc->wait_time(70);
  }
  vg4->window->clear();
  vg4->window->flush();

  /* remove space-key */
  vg4->input->key_remove(k_space);

  return VG_TRUE;
}


int main(int argc, char **argv) {
  struct s_game sgame;  /* private structure of the game */
  int audc_gameover, audc_bgmusic;  /* audio descriptors */
  int clnr, clmax, cli;
  VG_BOOL dowait;

  (void)argc; (void)argv;

  /* initialize game-structure */
  memset(&sgame, 0, sizeof(sgame));

  /* initialize and open window */
  if (!VG_init("VgaGames4 tutorial 7")) { exit(1); }
  if (!vg4->window->open(VG_WINDOW_SIZE_LOW, VG_WINDOW_SCALE_BEST)) { VG_dest(); exit(1); }
  vg4->window->getsize(&sgame.winw, &sgame.winh);

  /* set mouse grabbing to off */
  vg4->input->mouse_grabbing(VG_FALSE);

  /* open audio system */
  if (!vg4->audio->open(VG_AUDIO_FREQ_MEDIUM, VG_FALSE)) { VG_dest(); exit(1); }

  /* add object-collision functions */
  objcoll();

  /* create collision-tag */
  sgame.coll_tag = vg4->collision->create(NULL, 0, 0);

  /* load audio files */
  audc_gameover = vg4->audio->load("files/gameover.wav", 100, VG_AUDIO_VOLUME_SOUND);
  audc_bgmusic = vg4->audio->load("files/bgmusic.wav", 50, VG_AUDIO_VOLUME_MUSIC);

  /* set keys */
  /* quit with ALT+Q, not changeable, local key */
  if ((sgame.kref.k_quit_lalt = vg4->input->key_insert("Quit-LALT", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_quit_lalt, VG_INPUT_KBDCODE_LALT);
  if ((sgame.kref.k_quit_q = vg4->input->key_insert("Quit-Q", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_quit_q, VG_INPUT_KBDCODE_Q);
  /* system-menu with ESC, not changeable, local key */
  if ((sgame.kref.k_sysmenu = vg4->input->key_insert("System-menu", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_sysmenu, VG_INPUT_KBDCODE_ESCAPE);
  /* pause with P, changeable, local key */
  if ((sgame.kref.k_pause = vg4->input->key_insert("Pause", VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_pause, VG_INPUT_KBDCODE_P);
  /* move-up with cursor-key UP, changeable, network key */
  if ((sgame.kref.k_up = vg4->input->key_insert("Move up", VG_TRUE, VG_TRUE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_up, VG_INPUT_KBDCODE_UCURS);
  vg4->input->key_setgc(sgame.kref.k_up, 0, VG_INPUT_GCAXIS_RIGHTY_UP);
  /* move-down with cursor-key DOWN, changeable, network key */
  if ((sgame.kref.k_down = vg4->input->key_insert("Move down", VG_TRUE, VG_TRUE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_down, VG_INPUT_KBDCODE_DCURS);
  vg4->input->key_setgc(sgame.kref.k_down, 0, VG_INPUT_GCAXIS_RIGHTY_DOWN);

  /* show help */
  if (!show_help()) { VG_dest(); exit(1); }

  /* connect to network-server */
  if ((clnr = connect_to_nwserver(2, 2)) == 0) { VG_dest(); exit(1); }
  clmax = vg4->nw->numberofclients(NULL);
  if (clmax != 2) { fprintf(stderr, "There must be exact 2 players.\n"); VG_dest(); exit(1); }

  /* create object-instances */
  for (cli = 1; cli <= clmax; cli++) {  /* a player for each client */
    if (objnew_PLAYER(&sgame, cli) == 0) { VG_dest(); exit(1); }
  }
  if (objnew_BALL(&sgame) == 0) { VG_dest(); exit(1); }
  if (objnew_BORDER(&sgame, "top") == 0) { VG_dest(); exit(1); }
  if (objnew_BORDER(&sgame, "bottom") == 0) { VG_dest(); exit(1); }

  /* play background music looping */
  vg4->audio->play(audc_bgmusic, VG_TRUE, VG_TRUE);

  /* game loop */
  for (;;) {
    /* retrieve input-events from network-server */
    if (!vg4->nw->update(&dowait, NULL)) { goto endgame; }

    /* check if own client is still connected */
    if (!vg4->nw->is_connected(clnr)) { goto endgame; }

    /* quit? */
    if (vg4->input->key_newpressed(sgame.kref.k_quit_q) && vg4->input->key_pressed(sgame.kref.k_quit_lalt)) { goto endgame; }

    /* pause? */
    if (vg4->input->key_newpressed(sgame.kref.k_pause)) { vg4->nw->pause(); }

    /* system-menu? */
    if (vg4->input->key_newpressed(sgame.kref.k_sysmenu)) {
      struct VG_Hash *hvar = vg4->hash->create();
      vg4->nw->pause();  /* send pause-request to suspend the game */
      vg4->hash->setstr(hvar, "top:title", "System-menu");
      vg4->hash->setstr(hvar, "volume:top:title", "Set audio volumes");
      vg4->hash->setstr(hvar, "keydef:top:title", "Key Redefinition");
      vg4->hash->setstr(hvar, "keydef:press:title", "Press key");
      vg4->audio->suspend(VG_TRUE);
      if (!vg4->dialog->sysmenu("files/canvas", NULL, hvar, &sgame, NULL, NULL, NULL)) { goto endgame; }
      vg4->audio->suspend(VG_FALSE);
      vg4->hash->destroy(hvar);
    }

    /* call f_run() of all object-instances */
    if (!vg4->object->call_run(&sgame)) {
      if (sgame.winner != PLY_NOBODY) {
        /* show message, play audio and wait for its end */
        struct VG_Image *imgp;
        if (sgame.winner == sgame.ply_local) {  /* local player wins */
          imgp = vg4->font->totext("You win", "[fgcolor=0x00ff00]", NULL, NULL, NULL);
        } else {  /* local player looses */
          imgp = vg4->font->totext("You loose", "[fgcolor=0xff0000]", NULL, NULL, NULL);
        }
        vg4->window->copy(imgp, NULL, NULL);
        vg4->image->destroy(imgp);
        vg4->audio->stop(audc_bgmusic, VG_FALSE);
        vg4->audio->play(audc_gameover, VG_FALSE, VG_FALSE);
        while (vg4->audio->is_playing(audc_gameover, NULL)) { 
          vg4->window->flush();
          vg4->misc->wait_time(40);
        }
      }
      goto endgame;
    }

    /* clear window and draw background */
    vg4->window->clear();
    vg4->window->fill(vg4->misc->colorbrightness(VG_COLOR_BLUE, 50));

    /* call f_draw() of all object-instances */
    vg4->object->call_draw(&sgame);

    /* flush contents to window and wait */
    vg4->window->flush();
    if (dowait) { vg4->misc->wait_time(40); }
  }

endgame:
  /* destroy and exit */
  VG_dest();
  exit(0);
}
