/* select a sunnyboy from a chain and inform the other clients */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames4.h>

VG_BOOL select_sb(char *[], int, VG_BOOL);

static VG_BOOL exec_canvas(char *, size_t, char *[], int);
static void show_info(const char *);


/* execute canvas to select sunnyboy-file
 * @param sbname   for returning sunnyboy-name
 * @param sbsize   sizeof(sbname)
 * @param sbnames  array of char-pointers with already selected sunnyboys
 * @param clmax    number of array-elements (equal to number of clients)
 * @return VG_TRUE = OK or VG_FALSE = got exit-request
 */
static VG_BOOL
exec_canvas(char *sbname, size_t sbsize, char *sbnames[], int clmax)
{
  const char *chain_name = "sb_chain";
  struct VG_Canvas *cvas;
  const char *selname;
  int icl;

  if (sbname == NULL || sbsize == 0) { return VG_FALSE; }
  if (sbnames == NULL || clmax < 1) { return VG_FALSE; }

  *sbname = '\0';

  /* load canvas */
  cvas = vg4->canvas->load("select_sb.cvas", NULL);
  if (cvas == NULL) { return VG_FALSE; }

  /* disable chain-elements of already set sunnyboy-names */
  for (icl = 1; icl <= clmax; icl++) {
    if (sbnames[icl - 1] != NULL) {
      vg4->canvas->chain_disable(cvas, chain_name, sbnames[icl - 1], VG_TRUE);
    }
  }

  for (;;) {
    /* execute canvas */
    if (!vg4->canvas->exec(cvas, NULL, &selname)) { return VG_FALSE; }
    if (selname == NULL) { break; }  /* cancelled via escape-key or cancel-button */

    /* act according to selection */

    if (*selname == '\0') {  /* no selection, but return-key pressed */
      /* no action, re-execute canvas */
      ;

    } else if (strcmp(selname, chain_name) == 0) {  /* item selected from chain */
      const char *chainkey = vg4->canvas->chain_get_activated(cvas, selname);
      if (chainkey != NULL) {
        snprintf(sbname, sbsize, "%s", chainkey);
        break;
      }
    }
  }

  vg4->canvas->destroy(cvas);

  if (*sbname == '\0') { return VG_FALSE; }
  return VG_TRUE;
}


/* show info */
static void
show_info(const char *text)
{
  struct VG_Image *img;

  if (text == NULL || *text == '\0') { return; }

  img = vg4->font->totext(text, NULL, NULL, NULL, NULL);
  vg4->window->clear();
  vg4->window->copy(img, NULL, NULL);
  vg4->window->flush();
  vg4->image->destroy(img);
}


/* select sunnyboy, get selected sunnyboys of all clients
 * @param sbnames    array of char-pointers for returning the sunnyboy-names
 * @param clmax      number of array-elements (equal to number of clients)
 * @param exclusive  whether sunnyboy-selection is exclusive (each client must have a different sunnyboy)
 * @return VG_TRUE = OK or VG_FALSE = got exit-request
 *
 * The receiving part of this function is a form of a replacement of vg4->nw->xdata_allclients_recv().
 */
VG_BOOL
select_sb(char *sbnames[], int clmax, VG_BOOL exclusive)
{
  char *ex_data, sbname[32];
  size_t ex_size;
  int ex_clnr, clnr, icl;

  /* sbnames shall be an array of clmax elements */
  if (sbnames == NULL || clmax < 1) { return VG_FALSE; }
  clnr = vg4->nw->local_clnr();
  if (exclusive && clnr == 0) { return VG_FALSE; }

  vg4->nw->xdata_retag();

reselect:
  /* select sunnyboy */
  if (!exec_canvas(sbname, sizeof(sbname), sbnames, clmax)) { return VG_FALSE; }

  /* show info */
  show_info("Receiving data ...");

  /* send selected sunnyboy-name to network-server */
  if (!vg4->nw->xdata_send(sbname, strlen(sbname) + 1)) { return VG_FALSE; }

  /* receive data from network-server */
  for (;;) {
    if (!vg4->nw->xdata_recv(&ex_data, &ex_size, &ex_clnr)) { return VG_FALSE; }

    if (ex_size > 0) {  /* sunnyboy-name received */
      if (sbnames[ex_clnr - 1] == NULL) {  /* set sunnyboy-name of client-number ex_clnr */
        if (exclusive) {  /* selection is exclusive */
          /* check if any client already owns the received sunnyboy-name */
          for (icl = 1; icl <= clmax; icl++) {
            if (sbnames[icl - 1] != NULL && strcmp(sbnames[icl - 1], ex_data) == 0) { break; }
          }
          if (icl > clmax) { sbnames[ex_clnr - 1] = ex_data; ex_data = NULL; }  /* no, set it */
        } else {  /* not exclusive */
          sbnames[ex_clnr - 1] = ex_data;
          ex_data = NULL;
        }
      }
      if (ex_data != NULL) { free(ex_data); }

    } else {  /* no data received */
      /* check if all connected clients have their sunnyboy-name */
      for (icl = 1; icl <= clmax; icl++) {
        if (sbnames[icl - 1] == NULL && vg4->nw->is_connected(icl)) { break; }
      }
      if (icl > clmax) { break; }  /* yes, we got all information */

      if (exclusive) {  /* selection is exclusive */
        /* check if another client has already my sunnyboy-selection */
        if (sbnames[clnr - 1] == NULL) {  /* my sunnyboy is not yet set */
          for (icl = 1; icl <= clmax; icl++) {
            if (sbnames[icl - 1] != NULL && strcmp(sbnames[icl - 1], sbname) == 0) { break; }
          }
          if (icl <= clmax) {  /* yes, so select another sunnyboy */
            show_info("Selected sunnyboy is no more available");
            sleep(3);
            vg4->window->clear();
            goto reselect;
          }
        }
      }

      /* window-flush and wait */
      vg4->window->flush();
      vg4->misc->wait_time(100);
    }
  }

  return VG_TRUE;
}
