#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames4.h>

extern VG_BOOL connect_to_nwserver(int, int *, int *);
extern VG_BOOL select_sb(char *[], int, VG_BOOL);

/* max. number of clients */
#define CLMAX 4

/* keys */
struct {
  int k_quit;
  int k_pause;
  int k_zdata;
  int k_nw_up;
  int k_nw_down;
  int k_nw_right;
  int k_nw_left;
} kref;

/* players (clients) */
static struct {
  struct VG_Image *img;     /* image */
  struct VG_Rect rect;      /* position */
  struct VG_Image *txtimg;  /* text-image (additional-data) */
  int txtshow;              /* counter for showing text-image */
} player[CLMAX], *playerp;


int main(int argc, char **argv) {
  VG_BOOL dowait;
  int winw, winh;
  int clnr, clmax, icl;
  char buf[128];
  struct VG_NwZdata nw_zdata;

  (void)argc; (void)argv;

  /* open window */
  if (!VG_init("test")) { exit(1); }
  if (!vg4->window->open(VG_WINDOW_SIZE_LOW, VG_WINDOW_SCALE_NONE)) { VG_dest(); exit(1); }
  vg4->window->getsize(&winw, &winh);
  vg4->input->mouse_grabbing(VG_FALSE);

  /* connect to network server, get client-number (clnr) and number of clients (clmax) */
  if (!connect_to_nwserver(CLMAX, &clnr, &clmax)) { VG_dest(); exit(1); }

  /* set keys */
  /* quit: Q (local key) */
  if ((kref.k_quit = vg4->input->key_insert("Quit", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_quit, VG_INPUT_KBDCODE_Q);
  /* pause: P (local key) */
  if ((kref.k_pause = vg4->input->key_insert("Pause", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_pause, VG_INPUT_KBDCODE_P);
  /* send additional-data: space (local key) */
  if ((kref.k_zdata = vg4->input->key_insert("Additional-data", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_zdata, VG_INPUT_KBDCODE_SPACE);
  /* move up: upper cursor (network-key) */
  if ((kref.k_nw_up = vg4->input->key_insert("Move Up", VG_FALSE, VG_TRUE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_nw_up, VG_INPUT_KBDCODE_UCURS);
  /* move down: lower cursor (network-key) */
  if ((kref.k_nw_down = vg4->input->key_insert("Move Down", VG_FALSE, VG_TRUE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_nw_down, VG_INPUT_KBDCODE_DCURS);
  /* move right: right cursor (network-key) */
  if ((kref.k_nw_right = vg4->input->key_insert("Move Right", VG_FALSE, VG_TRUE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_nw_right, VG_INPUT_KBDCODE_RCURS);
  /* move left: left cursor (network-key) */
  if ((kref.k_nw_left = vg4->input->key_insert("Move Left", VG_FALSE, VG_TRUE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_nw_left, VG_INPUT_KBDCODE_LCURS);

  /* initialize players */
  { char *sbnames[clmax];

    /* select and get selected sunnyboys */
    for (icl = 1; icl <= clmax; icl++) { sbnames[icl - 1] = NULL; }
    if (!select_sb(sbnames, clmax, VG_TRUE)) { VG_dest(); exit(1); }

    /* set players */
    for (icl = 1; icl <= clmax; icl++) {
      if (sbnames[icl - 1] == NULL) { continue; }  /* disconnected */
      playerp = &player[icl - 1];
      /* load image */
      snprintf(buf, sizeof(buf), "images/%s.bmp", sbnames[icl - 1]);
      free(sbnames[icl - 1]);
      playerp->img = vg4->image->load(buf);
      if (playerp->img == NULL) { VG_dest(); exit(1); }
      /* set position and size */
      playerp->rect.x = (winw * icl / clmax) - (winw / clmax / 2);
      playerp->rect.y = winh / 2;
      vg4->image->getsize(playerp->img, NULL, &playerp->rect.w, &playerp->rect.h);
      /* no additional-data */
      playerp->txtimg = NULL;
      playerp->txtshow = 0;
    }
  }

  /* Gameloop */
  for (;;) {
    if (!vg4->nw->update(&dowait, &nw_zdata)) { goto go_end; }
    if (!vg4->nw->is_connected(clnr)) { goto go_end; }  /* i am still connected? */

    /* quit (local key)? */
    if (vg4->input->key_newpressed(kref.k_quit)) { goto go_end; }

    /* pause requested (local key)? */
    if (vg4->input->key_newpressed(kref.k_pause)) { vg4->nw->pause(); }

    /* send additional-data (local key)? */
    if (vg4->input->key_newpressed(kref.k_zdata)) {
      /* create colored string */
      struct VG_NwZdata zdata_put;
      snprintf(zdata_put.data, sizeof(zdata_put.data),
               "%%{txt[bgcolor=0x888888]: Hi, i am %s %%}", vg4->nw->get_clientname(clnr));
      zdata_put.size = strlen(zdata_put.data) + 1;  /* including null-termination */
      vg4->nw->put_zdata(&zdata_put);
    }

    /* check for received additional-data */
    if (nw_zdata.size > 0) {
      playerp = &player[nw_zdata.clnr - 1];
      /* if still a previous text-image is present, destroy it */
      if (playerp->txtimg != NULL) { vg4->image->destroy(playerp->txtimg); }
      /* create text-image from received null-terminated string */
      playerp->txtimg = vg4->font->totext(nw_zdata.data, NULL, NULL, NULL, NULL);
      playerp->txtshow = 60;  /* number of game-loops to show text-image */
    }

    /* clear window */
    vg4->window->clear();

    /* do actions with each client */
    for (icl = 1; icl <= clmax; icl++) {
      if (!vg4->nw->is_connected(icl)) { continue; }

      playerp = &player[icl - 1];

      /* check for network-keys */
      /* move up */
      if (vg4->nw->key_pressed(icl, kref.k_nw_up)) {
        playerp->rect.y--;
        if (playerp->rect.y < 0) { playerp->rect.y = 0; }
      }
      /* move down */
      if (vg4->nw->key_pressed(icl, kref.k_nw_down)) {
        playerp->rect.y++;
        if (playerp->rect.y > winh - playerp->rect.h) { playerp->rect.y = winh - playerp->rect.h; }
      }
      /* move right */
      if (vg4->nw->key_pressed(icl, kref.k_nw_right)) {
        playerp->rect.x++;
        if (playerp->rect.x > winw - playerp->rect.w) { playerp->rect.x = winw - playerp->rect.w; }
      }
      /* move left */
      if (vg4->nw->key_pressed(icl, kref.k_nw_left)) {
        playerp->rect.x--;
        if (playerp->rect.x < 0) { playerp->rect.x = 0; }
      }

      /* draw client */
      { struct VG_Position posi;
        posi.pos = VG_POS_UPPER_LEFT;
        posi.x = playerp->rect.x;
        posi.y = playerp->rect.y;
        vg4->window->copy(playerp->img, &posi, NULL);
      }

      /* draw client's text-image if not NULL */
      if (playerp->txtimg != NULL) {
        struct VG_Position posi;
        posi.pos = VG_POS_CENTERED;
        posi.x = playerp->rect.x + playerp->rect.w / 2;
        posi.y = playerp->rect.y + playerp->rect.h / 2;
        vg4->window->copy(playerp->txtimg, &posi, NULL);
        if (--playerp->txtshow == 0) {  /* destroy text-image */
          vg4->image->destroy(playerp->txtimg);
          playerp->txtimg = NULL;
        }
      }
    }

    /* flush contents to window, and wait if not delayed */
    vg4->window->flush();
    if (dowait) { vg4->misc->wait_time(50); }
  }

go_end:
  VG_dest();
  exit(0);
}
