/* Copyright 2022-2026 Kurt Nienhaus
 *
 * This file is part of VgaGames.
 * VgaGames is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "misc.h"

void init_misc_utf8(void);

static int misc_utf8_next(const char *, size_t, int *);
static int misc_utf8_from_codepoint(int, char *);
static int misc_utf8_toupper(int);
static int misc_utf8_tolower(int);


/* set functions */
void
init_misc_utf8(void)
{
  vg4->misc->utf8_next = misc_utf8_next;
  vg4->misc->utf8_from_codepoint = misc_utf8_from_codepoint;
  vg4->misc->utf8_toupper = misc_utf8_toupper;
  vg4->misc->utf8_tolower = misc_utf8_tolower;
} /* Ende init_misc_utf8 */


/* misc_utf8_next:
 * find next UTF8-character
 * @param string  string
 * @param size    length of string
 * @param codept  for returning Unicode-codepoint of UTF8-character, if not NULL
 * @return   how many bytes UTF8-character needs
 *           or 0 = end
 *           or -1 = invalid character found
 *
 * Example:
 *   const char *string = "Word";
 *   const char *strptr;
 *   size_t strsize;
 *   int no_bytes, codept;
 *
 *   strptr = string;
 *   strsize = strlen(strptr);
 *   for (;;) {
 *     no_bytes = vg4->misc->utf8_next(strptr, strsize, &codept);
 *     if (no_bytes <= 0) { break; }
 *     printf("UTF8-character: codept = %d, number of bytes = %d: %.*s\n", codept, no_bytes, no_bytes, strptr);
 *     strptr += no_bytes;
 *     strsize -= no_bytes;
 *   }
 */
static int
misc_utf8_next(const char *string, size_t size, int *codept)
{
  return SML3_utf8_next(string, size, codept);
} /* Ende misc_utf8_next */


/* misc_utf8_from_codepoint:
 * set UTF8-character from Unicode-codepoint
 * @param codept  Unicode-codepoint
 * @param cbuf    for returning UTF8-character, if not NULL,
 *                (will not be null-terminated),
 *                should have at least 6 bytes
 * @return   of how many bytes UTF8-character consists, or 0 = invalid
 *
 * Example:
 *   char cbuf[8];
 *   int no_bytes, codept;
 *
 *   vg4->misc->utf8_next("X", sizeof("X"), &codept);  // get codepoint of X
 *   no_bytes = vg4->misc->utf8_from_codepoint(codept, cbuf);
 *   printf("UTF8-character: codept = %d, number of bytes = %d: %.*s\n", codept, no_bytes, no_bytes, cbuf);
 */
static int
misc_utf8_from_codepoint(int codept, char *cbuf)
{
  return SML3_utf8_from_codepoint(codept, cbuf);
} /* Ende misc_utf8_from_codepoint */


/* misc_utf8_toupper:
 * convert UTF8-character to uppercase
 * @param codept  Unicode-codepoint
 * @return   uppercase-character or unchanged character
 */
static int
misc_utf8_toupper(int codept)
{
  return SML3_utf8_toupper(codept);
} /* Ende misc_utf8_toupper */


/* misc_utf8_tolower:
 * convert UTF8-character to lowercase
 * @param codept  Unicode-codepoint
 * @return   lowercase-character or unchanged character
 */
static int
misc_utf8_tolower(int codept)
{
  return SML3_utf8_tolower(codept);
} /* Ende misc_utf8_tolower */
