/* Copyright 2022-2026 Kurt Nienhaus
 *
 * This file is part of VgaGames.
 * VgaGames is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "image.h"

VG_BOOL vg4_rect_overlap(const struct VG_Position *, struct VG_Rect *, struct VG_Rect *);
VG_BOOL vg4_attrpixel_is_default(const struct VG_ImagecopyAttrPixel *);
VG_BOOL vg4_attrimage_is_default(const struct VG_ImagecopyAttrImage *);


/* vg4_rect_overlap:
 * calculate overlapping part of source and destination rectangle
 * @param dst_pos   position on destination
 * @param src_rect  source rectangle,
 *                  - delivery value: rectangle of full source
 *                  - return value:   overlapping rectangle of source
 * @param dst_rect  destination rectangle,
 *                  - delivery value: rectangle of full destination
 *                  - return value:   overlapping rectangle of destination
 * @return  VG_BOOL:
 *          - VG_TRUE: overlapping occured
 *          - VG_FALSE: no overlapping occured
 */
VG_BOOL
vg4_rect_overlap(const struct VG_Position *dst_pos, struct VG_Rect *src_rect, struct VG_Rect *dst_rect)
{
  if (dst_pos == NULL || src_rect == NULL || dst_rect == NULL) { return VG_FALSE; }

  switch (dst_pos->pos) {
    case VG_POS_UPPER_LEFT:
      dst_rect->x = dst_pos->x;
      dst_rect->y = dst_pos->y;
      break;
    case VG_POS_UPPER_RIGHT:
      dst_rect->x = dst_pos->x - (src_rect->w - 1);
      dst_rect->y = dst_pos->y;
      break;
    case VG_POS_LOWER_LEFT:
      dst_rect->x = dst_pos->x;
      dst_rect->y = dst_pos->y - (src_rect->h - 1);
      break;
    case VG_POS_LOWER_RIGHT:
      dst_rect->x = dst_pos->x - (src_rect->w - 1);
      dst_rect->y = dst_pos->y - (src_rect->h - 1);
      break;
    case VG_POS_CENTERED:
      dst_rect->x = dst_pos->x - src_rect->w / 2;
      dst_rect->y = dst_pos->y - src_rect->h / 2;
      break;
    default:
      dst_rect->x = dst_pos->x;
      dst_rect->y = dst_pos->y;
      break;
  }

  if (dst_rect->x < 0) { src_rect->x -= dst_rect->x; src_rect->w += dst_rect->x; dst_rect->x = 0; }
  if (src_rect->w <= 0) { return VG_FALSE; }
  if (dst_rect->x + src_rect->w > dst_rect->w) { src_rect->w -= (dst_rect->x + src_rect->w - dst_rect->w); }
  if (src_rect->w <= 0) { return VG_FALSE; }
  if (dst_rect->y < 0) { src_rect->y -= dst_rect->y; src_rect->h += dst_rect->y; dst_rect->y = 0; }
  if (src_rect->h <= 0) { return VG_FALSE; }
  if (dst_rect->y + src_rect->h > dst_rect->h) { src_rect->h -= (dst_rect->y + src_rect->h - dst_rect->h); }
  if (src_rect->h <= 0) { return VG_FALSE; }

  dst_rect->w = src_rect->w;
  dst_rect->h = src_rect->h;

  return VG_TRUE;
} /* Ende vg4_rect_overlap */


/* vg4_attrpixel_is_default:
 * return whether pixel-modifying part of image-copy attributes has default values
 * @param iattr_pixel  pixel-modifying part of image-copy attributes
 * @return  VG_BOOL:
 *          - VG_TRUE:  is default
 *          - VG_FALSE: is not default
 */
VG_BOOL
vg4_attrpixel_is_default(const struct VG_ImagecopyAttrPixel *iattr_pixel)
{
  if (iattr_pixel == NULL) { return VG_TRUE; }

  if (iattr_pixel->opaqueness < 100 && iattr_pixel->opaqueness >= 0) { return VG_FALSE; }
  if (iattr_pixel->brightness != 100 && iattr_pixel->brightness >= 0 && iattr_pixel->brightness <= VG_MAX_BRIGHTNESS) { return VG_FALSE; }
  if (iattr_pixel->colorize_percent > 0 && iattr_pixel->colorize_percent <= 100) {
    if (iattr_pixel->colorize_color != VG_COLOR_TRANSPARENT && iattr_pixel->colorize_color >= 0) { return VG_FALSE; }
  }
  if (iattr_pixel->pixelcolor != VG_PIXELCOLOR_NOP) { return VG_FALSE; }
  if (iattr_pixel->cbf != NULL) { return VG_FALSE; }

  return VG_TRUE;
} /* Ende vg4_attrpixel_is_default */


/* vg4_attrimage_is_default:
 * return whether image-modifying part of image-copy attributes has default values
 * @param iattr_image  image-modifying part of image-copy attributes
 * @return  VG_BOOL:
 *          - VG_TRUE:  is default
 *          - VG_FALSE: is not default
 */
VG_BOOL
vg4_attrimage_is_default(const struct VG_ImagecopyAttrImage *iattr_image)
{
  if (iattr_image == NULL) { return VG_TRUE; }

  if (!iattr_image->zoom_ispercent || iattr_image->zoom_width != 100 || iattr_image->zoom_height != 100) { return VG_FALSE; }
  if (iattr_image->rotate % 360 != 0 || iattr_image->flip != VG_AXE_NONE) { return VG_FALSE; }

  return VG_TRUE;
} /* Ende vg4_attrimage_is_default */
