/* Copyright 2022-2026 Kurt Nienhaus
 *
 * This file is part of VgaGames.
 * VgaGames is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "vgi.h"

VG_BOOL init_iolib(void);
void dest_iolib(void);


/* init_iolib:
 * call init-function of selected or default iolib-library
 * @return  VG_TRUE = OK or VG_FALSE = error
 */
VG_BOOL
init_iolib(void)
{
  const char *io_library;
  const char *tag;

  if (vg4data.iolib.stptr != NULL && *vg4data.iolib.tag != '\0') {
    outerr("initializing io-library: is already initialized with %s", vg4data.iolib.tag);
    return VG_FALSE;
  }

  io_library = getenv("VG_IO_LIBRARY");
  if (io_library != NULL && *io_library == '\0') { io_library = NULL; }  /* use first found io-library */

  if (io_library != NULL) {
    pinfo("VG_IO_LIBRARY set as \"%s\".\n", io_library);
  } else {
    pinfo("No VG_IO_LIBRARY set, loading first found io-library.\n");
  }

#ifdef IOLIB_LIB_SDL3
  tag = "SDL3";
  pinfo("Found io-library: %s\n", tag);
  if (io_library == NULL || strcmp(io_library, tag) == 0) {  /* not specified or SDL3 */
    extern int init_iolib_sdl3(void);
    if (!init_iolib_sdl3()) { return VG_FALSE; }
    snprintf(vg4data.iolib.tag, sizeof(vg4data.iolib.tag), "%s", tag);
    io_library = tag;
  }
#endif

#ifdef IOLIB_LIB_SDL2
  tag = "SDL2";
  pinfo("Found io-library: %s\n", tag);
  if (io_library == NULL || strcmp(io_library, tag) == 0) {  /* not specified or SDL2 */
    extern int init_iolib_sdl2(void);
    if (!init_iolib_sdl2()) { return VG_FALSE; }
    snprintf(vg4data.iolib.tag, sizeof(vg4data.iolib.tag), "%s", tag);
    io_library = tag;
  }
#endif

  /* FIXME: more io-libraries */

  if (vg4data.iolib.stptr == NULL) {
    if (io_library == NULL) {
      outerr("initializing io-library: cannot find any io-library");
    } else {
      outerr("initializing io-library: cannot find %s", io_library);
    }
    return VG_FALSE;
  }

  return VG_TRUE;
} /* Ende init_iolib */


/* dest_iolib:
 * call dest-function of loaded iolib-library
 */
void
dest_iolib(void)
{
  if (*vg4data.iolib.tag == '\0') { return; }

#ifdef IOLIB_LIB_SDL2
  if (strcmp(vg4data.iolib.tag, "SDL2") == 0) {
    extern void dest_iolib_sdl2(void);
    dest_iolib_sdl2();
  }
#endif

#ifdef IOLIB_LIB_SDL3
  if (strcmp(vg4data.iolib.tag, "SDL3") == 0) {
    extern void dest_iolib_sdl3(void);
    dest_iolib_sdl3();
  }
#endif
  /* FIXME: more io-libraries */

  *vg4data.iolib.tag = '\0';
} /* Ende dest_iolib */
