/* Copyright 2022-2026 Kurt Nienhaus
 *
 * This file is part of VgaGames.
 * VgaGames is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "dialog.h"

void init_dialog_winbright(void);

static VG_BOOL dialog_winbright(const char *, const struct VG_Position *, struct VG_Hash *);


/* set functions */
void
init_dialog_winbright(void)
{
  vg4->dialog->winbright = dialog_winbright;
} /* Ende init_dialog_winbright */


/* dialog_winbright:
 * open a canvas-dialog for setting common window brightness
 * @param cvasdir     directory to load canvas from, or NULL = system-canvas
 * @param posdst      canvas-position on window, or NULL = centered
 * @param hvar        hash with variable-values for text control-commands "var", or NULL
 *                      hash-format:
 *                       - key:   variable name
 *                       - value: variable value
 *                    (see below)
 * @return  VG_TRUE = OK or VG_FALSE = exit-request
 *
 * The canvas-name must be:
 *  - winbright.top.cvas: for setting common window brightness
 *    - with items:
 *      - [CV-TEXT]   with name "title": text for title
 *      - [CV-BAR]    with name "bar-brightness": for setting brightness
 *      - [CV-BUTTON] with name "reset": for resetting to default value
 *      - [CV-BUTTON] with name "apply": for applying changes
 *      - [CV-BUTTON] as a cancel-button
 *    - uses following variables:
 *      - key: "title", value: <title of canvas>
 *    - hvar should contain:
 *      - key: "top:title", value: <title of canvas>
 */
static VG_BOOL
dialog_winbright(const char *cvasdir, const struct VG_Position *posdst, struct VG_Hash *hvar)
{
  const char *cvprefix = "winbright";
  const char *cvfile_top = "winbright.top.cvas";
  const char *cvname_bar_brightness = "bar-brightness";
  const char *cvname_reset = "reset";
  const char *cvname_apply = "apply";
  const char *cvvar_ovdpc = "ovdpc";
  struct VG_Hash *hvar_top;
  char cvaspath[512];
  VG_BOOL retw;
  struct VG_Canvas *cvas_top;
  const char *selname;
  int wbright, imod;

  hvar_top = NULL;

  /* get brightness */
  wbright = vg4->window->getbrightness();
  imod = wbright * 1000 / VG_MAX_BRIGHTNESS % 10;
  wbright = wbright * 100 / VG_MAX_BRIGHTNESS + imod / 5;

  /* set variables and load canvas */

  /* top-canvas */
  hvar_top = vg4->hash->create();
  dialog_transfer_variables(hvar, hvar_top, "top");
  { char bf1[32];
    int wbrightdef = 100;
    imod = wbrightdef * 1000 / VG_MAX_BRIGHTNESS % 10;
    wbrightdef = wbrightdef * 100 / VG_MAX_BRIGHTNESS + imod / 5;
    snprintf(bf1, sizeof(bf1), "%d", wbrightdef + 1);
    /* set overdrive-percent */
    vg4->hash->setstr(hvar_top, cvvar_ovdpc, bf1);
  }
  dialog_cvasfile(cvprefix, cvfile_top, cvasdir, cvaspath, sizeof(cvaspath));
  cvas_top = vg4->canvas->load(cvaspath, hvar_top);
  if (cvas_top == NULL) {
    outerr("%s [at cvasdir=%s] not found", cvfile_top, (cvasdir == NULL ? "(null)" : cvasdir));
    return VG_TRUE;
  }
  vg4->canvas->bar_set_percent(cvas_top, cvname_bar_brightness, wbright);

  /* loop */

  retw = VG_TRUE;
  for (;;) {
    /* execute top-canvas */
    if (!vg4->canvas->exec(cvas_top, posdst, &selname)) { retw = VG_FALSE; break; }
    if (selname == NULL) { break; }  /* cancel */

    /* act according to selection */
    if (strcmp(selname, cvname_reset) == 0) {
      wbright = 100;
      imod = wbright * 1000 / VG_MAX_BRIGHTNESS % 10;
      wbright = wbright * 100 / VG_MAX_BRIGHTNESS + imod / 5;
      vg4->canvas->bar_set_percent(cvas_top, cvname_bar_brightness, wbright);

    } else if (strcmp(selname, cvname_apply) == 0) {
      wbright = vg4->canvas->bar_get_percent(cvas_top, cvname_bar_brightness);
      if (wbright < 5) { wbright = 5; }
      wbright = wbright * VG_MAX_BRIGHTNESS / 100;
      if (wbright >= 99 && wbright <= 101) { wbright = 100; }
      vg4->window->setbrightness(wbright);
      break;
    }
  }

  vg4->canvas->destroy(cvas_top);
  if (hvar_top != NULL) { vg4->hash->destroy(hvar_top); }

  return retw;
} /* Ende dialog_winbright */
