/* Copyright 2022-2026 Kurt Nienhaus
 *
 * This file is part of VgaGames.
 * VgaGames is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "dialog.h"

void init_dialog_text(void);

static VG_BOOL dialog_text(const char *, const char *, const struct VG_Position *, struct VG_Hash *);


/* set functions */
void
init_dialog_text(void)
{
  vg4->dialog->text = dialog_text;
} /* Ende init_dialog_text */


/* dialog_text:
 * open a canvas-dialog for showing a text
 * @param fsize       which canvas-file: "small", "medium" or "large"
 * @param cvasdir     directory to load canvas from, or NULL = system-canvas
 * @param posdst      canvas-position on window, or NULL = centered
 * @param hvar        hash with variable-values for text control-commands "var", or NULL
 *                      hash-format:
 *                       - key:   variable name
 *                       - value: variable value
 *                    (see below)
 * @return  VG_TRUE = OK or VG_FALSE = exit-request
 *
 * The canvas-name must be:
 *  - text.small.cvas: for showing the text (small canvas)
 *    text.medium.cvas: for showing the text (medium canvas)
 *    text.large.cvas: for showing the text (large canvas)
 *    - with items:
 *      - [CV-TEXT]   with name "title": text for title
 *      - [CV-TEXT]   with name "text": the text
 *      - [CV-BUTTON] with name "close": for returning
 *    - uses following variables:
 *      - key: "title", value: <title of canvas>
 *      - key: "text", value: <the text>
 *      - key: "velo", value: <moving velocitas in pixels>
 *      - key: "font", value: <font, e.g. sys:low>
 *      - key: "fgcolor", value: <text color>
 *    - hvar should contain:
 *      - key: "top:title", value: <title of canvas>
 *      - key: "top:text", value: <the text>
 *      - key: "top:velo", value: <moving velocitas in pixels>
 *      - key: "top:font", value: <font, e.g. sys:low>
 *      - key: "top:fgcolor", value: <text color>
 */
static VG_BOOL
dialog_text(const char *fsize, const char *cvasdir, const struct VG_Position *posdst, struct VG_Hash *hvar)
{
  const char *cvprefix = "text";
  const char *cvfile_small = "text.small.cvas";
  const char *cvfile_medium = "text.medium.cvas";
  const char *cvfile_large = "text.large.cvas";
  const char *cvname_close = "close";
  struct VG_Hash *hvar_top;
  char cvaspath[512];
  VG_BOOL retw;
  struct VG_Canvas *cvas_top;
  const char *cvfile, *selname;

  hvar_top = NULL;

  /* set variables and load canvas */

  if (fsize == NULL) { fsize = "medium"; }
  if (strcmp(fsize, "small") == 0) {
    cvfile = cvfile_small;
  } else if (strcmp(fsize, "medium") == 0) {
    cvfile = cvfile_medium;
  } else if (strcmp(fsize, "large") == 0) {
    cvfile = cvfile_large;
  } else {
    cvfile = cvfile_medium;
  }

  /* canvas */
  hvar_top = vg4->hash->create();
  dialog_transfer_variables(hvar, hvar_top, "top");
  dialog_cvasfile(cvprefix, cvfile, cvasdir, cvaspath, sizeof(cvaspath));
  cvas_top = vg4->canvas->load(cvaspath, hvar_top);
  if (cvas_top == NULL) {
    outerr("%s [at cvasdir=%s] not found", cvfile, (cvasdir == NULL ? "(null)" : cvasdir));
    return VG_TRUE;
  }

  /* loop */

  retw = VG_TRUE;
  for (;;) {
    /* execute top-canvas */
    if (!vg4->canvas->exec(cvas_top, posdst, &selname)) { retw = VG_FALSE; break; }
    if (selname == NULL) { break; }  /* cancel */

    /* act according to selection */
    if (*selname == '\0' || strcmp(selname, cvname_close) == 0) { break; }
  }

  vg4->canvas->destroy(cvas_top);
  if (hvar_top != NULL) { vg4->hash->destroy(hvar_top); }

  return retw;
} /* Ende dialog_text */
