/* Copyright 2022-2026 Kurt Nienhaus
 *
 * This file is part of VgaGames.
 * VgaGames is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "dialog.h"

void init_dialog_exit(void);

static VG_BOOL dialog_exit(const char *, const struct VG_Position *, struct VG_Hash *);


/* set functions */
void
init_dialog_exit(void)
{
  vg4->dialog->exit = dialog_exit;
} /* Ende init_dialog_exit */


/* dialog_exit:
 * open a canvas-dialog to confirm exiting game
 * @param cvasdir     directory to load canvas from, or NULL = system-canvas
 * @param posdst      canvas-position on window, or NULL = centered
 * @param hvar        hash with variable-values for text control-commands "var", or NULL
 *                      hash-format:
 *                       - key:   variable name
 *                       - value: variable value
 *                    (see below)
 * @return  VG_TRUE = yes or VG_FALSE = no
 *
 * The canvas-name must be:
 *  - exit.top.cvas: for confirming
 *    - with items:
 *      - [CV-TEXT]   with name "title": text for title
 *      - [CV-BUTTON] with name "yes": yes
 *      - [CV-BUTTON] as a cancel-button
 *    - uses following variables:
 *      - key: "title", value: <title of canvas>
 *    - hvar should contain:
 *      - key: "top:title", value: <title of canvas>
 */
static VG_BOOL
dialog_exit(const char *cvasdir, const struct VG_Position *posdst, struct VG_Hash *hvar)
{
  const char *cvprefix = "exit";
  const char *cvfile_top = "exit.top.cvas";
  const char *cvname_yes = "yes";
  struct VG_Hash *hvar_top;
  char cvaspath[512];
  VG_BOOL retw;
  struct VG_Canvas *cvas_top;
  const char *selname;

  hvar_top = NULL;

  /* set variables and load canvas */

  /* top-canvas */
  hvar_top = vg4->hash->create();
  dialog_transfer_variables(hvar, hvar_top, "top");
  dialog_cvasfile(cvprefix, cvfile_top, cvasdir, cvaspath, sizeof(cvaspath));
  cvas_top = vg4->canvas->load(cvaspath, hvar_top);
  if (cvas_top == NULL) {
    outerr("%s [at cvasdir=%s] not found", cvfile_top, (cvasdir == NULL ? "(null)" : cvasdir));
    return VG_FALSE;
  }

  /* loop */

  retw = VG_FALSE;
  for (;;) {
    /* execute top-canvas */
    if (!vg4->canvas->exec(cvas_top, posdst, &selname)) { retw = VG_TRUE; break; }
    if (selname == NULL) { break; }  /* cancel */

    /* act according to selection */
    if (strcmp(selname, cvname_yes) == 0) { retw = VG_TRUE; break; }  /* exit */
  }

  vg4->canvas->destroy(cvas_top);
  if (hvar_top != NULL) { vg4->hash->destroy(hvar_top); }

  return retw;
} /* Ende dialog_exit */
