/* Copyright 2022-2026 Kurt Nienhaus
 *
 * This file is part of VgaGames.
 * VgaGames is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * VgaGames is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with VgaGames.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "dialog.h"

void init_dialog(void);
void dest_dialog(void);

void dialog_cvasfile(const char *, const char *, const char *, char *, size_t);
void dialog_transfer_variables(struct VG_Hash *, struct VG_Hash *, const char *);
VG_BOOL dialog_exec_subcvas(struct VG_Canvas *, struct VG_Canvas *, const char **);


/* set functions */
void
init_dialog(void)
{
  init_dialog_fsel();
  init_dialog_exit();
  init_dialog_wdsize();
  init_dialog_volume();
  init_dialog_keydef();
  init_dialog_sysmenu();
  init_dialog_network();
  init_dialog_text();
  init_dialog_winbright();
} /* Ende init_dialog */

void
dest_dialog(void)
{
  ;
} /* Ende dest_dialog */


/* set path for canvas file */
void
dialog_cvasfile(const char *cvprefix, const char *cvasfile, const char *cvasdir, char *retpath, size_t pathsize)
{
  const char *prefix = vg4_get_prefix(NULL);

  if (retpath == NULL || pathsize == 0) { return; }
  *retpath = '\0';
  if (cvasfile == NULL || *cvasfile == '\0') { return; }
  if (cvprefix == NULL || *cvprefix == '\0') { cvprefix = "."; }

  if (cvasdir == NULL || *cvasdir == '\0') {
    int wsize;
    vg4->window->getparameters(&wsize, NULL);
    if (wsize == VG_WINDOW_SIZE_LOW) {
      snprintf(retpath, pathsize, "%s/share/vgagames4/canvas/%s/%s/%s", prefix, VGI_SIZE_LOW, cvprefix, cvasfile);
    } else {
      snprintf(retpath, pathsize, "%s/share/vgagames4/canvas/%s/%s/%s", prefix, VGI_SIZE_HIGH, cvprefix, cvasfile);
    }
  } else {
    snprintf(retpath, pathsize, "%s/%s/%s", cvasdir, cvprefix, cvasfile);
    if (access(retpath, R_OK) < 0) {
      snprintf(retpath, pathsize, "%s/%s", cvasdir, cvasfile);
    }
  }
} /* Ende dialog_cvasfile */


/* transfer variables beginning with cvscope + ':' from hvar_from to hvar_to */
void
dialog_transfer_variables(struct VG_Hash *hvar_from, struct VG_Hash *hvar_to, const char *cvscope)
{
  void *vpos;
  const char *key, *val;
  size_t scoplen;

  if (hvar_from == NULL || hvar_to == NULL || cvscope == NULL || *cvscope == '\0') { return; }

  scoplen = strlen(cvscope);

  vpos = NULL;
  for (key = vg4->hash->list(hvar_from, &vpos); vpos != NULL; key = vg4->hash->list(hvar_from, &vpos)) {
    if (strncmp(key, cvscope, scoplen) == 0 && key[scoplen] == ':' && key[scoplen + 1] != '\0') {
      val = (const char *)vg4->hash->get(hvar_from, key, NULL);
      vg4->hash->setstr(hvar_to, key + scoplen + 1, val);
    }
  }
} /* Ende dialog_transfer_variables */


/* execute sub-canvas */
VG_BOOL
dialog_exec_subcvas(struct VG_Canvas *cvas_parent, struct VG_Canvas *cvas, const char **selname)
{
  struct VG_Position possub;
  struct VG_Image *wclone;
  struct VG_ImagecopyAttrPixel iattr_pixel;

  if (selname != NULL) { *selname = NULL; }
  if (cvas == NULL || selname == NULL) { return VG_TRUE; }

  VG_IMAGECOPY_ATTRPIXEL_DEFAULT(&iattr_pixel);
  iattr_pixel.brightness = 60;

  /* get current background */
  wclone = vg4->window->clone(NULL, NULL);

  /* get position for canvas and draw background */
  if (cvas_parent != NULL) {
    possub = vg4->canvas->subcanvas(cvas_parent, &iattr_pixel);
  } else {  /* no parent-canvas */
    vg4->window->getsize(&possub.x, &possub.y);
    possub.pos = VG_POS_CENTERED;
    possub.x /= 2;
    possub.y /= 2;
  }

  /* execute canvas */
  if (!vg4->canvas->exec(cvas, &possub, selname)) { return VG_FALSE; }

  /* restore background */
  vg4->window->clear();
  vg4->window->copy(wclone, NULL, NULL);
  vg4->image->destroy(wclone);

  return VG_TRUE;
} /* Ende dialog_exec_subcvas */
