/* canvas-type button
 *   [CV-BUTTON]
 *   name: <arbitrary unique name>
 *   position-itag: (position): <box-tag of an element in the main text>
 *   position-rect: (position): <rectangle (<x>+<w>,<y>+<h>) within the box-tag or the main text>
 *   disable: <whether is disabled: 1 = yes, 0 = no, or missing = enabled>
 *   is-cancel: <whether button is a cancel-button: 1 = yes, 0 = no, or missing = no>
 *   img-dfl: (default image):   [img:]<image file> or txt:<text file> or sprt:<sprite file> or missing
 *   img-act: (activated image): [img:]<image file> or txt:<text file> or sprt:<sprite file> or missing = img-dfl
 *   img-sel: (selected image):  [img:]<image file> or txt:<text file> or sprt:<sprite file> or missing = img-dfl
 *   orientation: <left|center|right>
 *   v-orientation: <top|center|bottom>
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "canvas.h"

static int type_no = 0;

struct cv_type {
  VG_BOOL is_cancel;          /* whether pressed button means Cancel instead of OK */
  int orientation;            /* one of VG_ORIENTATIONS */
  int v_orientation;          /* one of VG_ORIENTATIONS */
  struct vgi_imgcnt img_dfl;  /* default image, or NULL */
  struct vgi_imgcnt img_act;  /* activated image, or NULL = default image */
  struct vgi_imgcnt img_sel;  /* selected image, or NULL = default image */
};

void cvinit_button(struct vgi_canvas_inittype *, int);

static void * t_read_section(const struct VG_Canvas *, struct vgi_canvas_item *, const char *);
static void t_draw(struct vgi_canvas_data *, struct vgi_canvas_item *, const char **);
static void t_destroy(struct vgi_canvas_item *);
static void t_dump(FILE *, struct vgi_canvas_item *);


void
cvinit_button(struct vgi_canvas_inittype *cvinittype, int type)
{
  if (cvinittype == NULL) { return; }

  type_no = type;

  /* cvinit */
  cvinittype->mouse_opaque = VG_FALSE;
  cvinittype->uses_cursor = VG_FALSE;
  cvinittype->read_section = t_read_section;
  cvinittype->get_itag_rect = NULL;
  cvinittype->warp_mouse = NULL;
  cvinittype->pre_exec = NULL;
  cvinittype->draw = t_draw;
  cvinittype->destroy = t_destroy;
  cvinittype->dump = t_dump;
} /* Ende cvinit_button */


static void *
t_read_section(const struct VG_Canvas *cvas, struct vgi_canvas_item *cvitem, const char *dtptr)
{
  struct SML3_gummi gm1 = SML3_GUM_INITIALIZER;
  struct cv_type *cvtype;
  char bprefix[32], bval[256], *pval;

  if (cvas == NULL || cvitem == NULL || dtptr == NULL) { return NULL; }

  cvtype = SML3_calloc(1, sizeof(*cvtype));

  if (canvas_get_lineval(dtptr, "is-cancel", NULL, 0, NULL, 0, bval, sizeof(bval), NULL)) {
    vg4_intern_str_with_var(&gm1, bval, cvas->cve.hvar, NULL); pval = SML3_gumgetval(&gm1);
    DETECT_BOOLEAN(pval, cvtype->is_cancel);
  }

  if (canvas_get_lineval(dtptr, "img-dfl", NULL, 0, bprefix, sizeof(bprefix), bval, sizeof(bval), NULL)) {
    canvas_load_image(cvas, bprefix, bval, &cvtype->img_dfl);
  }

  if (canvas_get_lineval(dtptr, "img-act", NULL, 0, bprefix, sizeof(bprefix), bval, sizeof(bval), NULL)) {
    canvas_load_image(cvas, bprefix, bval, &cvtype->img_act);
  }

  if (canvas_get_lineval(dtptr, "img-sel", NULL, 0, bprefix, sizeof(bprefix), bval, sizeof(bval), NULL)) {
    canvas_load_image(cvas, bprefix, bval, &cvtype->img_sel);
  }

  if (canvas_get_lineval(dtptr, "orientation", NULL, 0, NULL, 0, bval, sizeof(bval), NULL)) {
    vg4_intern_str_with_var(&gm1, bval, cvas->cve.hvar, NULL); pval = SML3_gumgetval(&gm1);
    if (strcmp(pval, "left") == 0) {
      cvtype->orientation = VG_ORI_LEFT;
    } else if (strcmp(pval, "center") == 0) {
      cvtype->orientation = VG_ORI_CENTER;
    } else if (strcmp(pval, "right") == 0) {
      cvtype->orientation = VG_ORI_RIGHT;
    } else {
      cvtype->orientation = VG_ORI_CENTER;
    }
  }

  if (canvas_get_lineval(dtptr, "v-orientation", NULL, 0, NULL, 0, bval, sizeof(bval), NULL)) {
    vg4_intern_str_with_var(&gm1, bval, cvas->cve.hvar, NULL); pval = SML3_gumgetval(&gm1);
    if (strcmp(pval, "top") == 0) {
      cvtype->v_orientation = VG_ORI_LEFT;
    } else if (strcmp(pval, "center") == 0) {
      cvtype->v_orientation = VG_ORI_CENTER;
    } else if (strcmp(pval, "bottom") == 0) {
      cvtype->v_orientation = VG_ORI_RIGHT;
    } else {
      cvtype->v_orientation = VG_ORI_CENTER;
    }
  }

  SML3_gumdest(&gm1);

  return cvtype;
} /* Ende t_read_section */


static void
t_draw(struct vgi_canvas_data *cvdata, struct vgi_canvas_item *cvitem, const char **selname)
{
  struct cv_type *cvtype;

  if (cvdata == NULL || cvdata->cvas == NULL || cvitem == NULL || selname == NULL) { return; }
  if (cvitem->rimg == NULL) { return; }
  cvtype = (struct cv_type *)cvitem->vstrct;
  if (cvtype == NULL) { return; }

  /* draw */
  { struct VG_Image *imgp;
    struct VG_ImagecopyAttr iattr;
    struct VG_Position posb;
    VG_BOOL drawborder;

    imgp = NULL;
    drawborder = VG_FALSE;

    if (cvdata->hasfocus) {  /* has focus */
      if (VGI_CVKEY_PRESSED(cvdata->keys.k[CV_KEY_RETURN]) || VGI_CVKEY_PRESSED(cvdata->keys.m_left)) {
        imgp = vg4_intern_get_img_from_imgcnt(&cvtype->img_sel, &iattr);
      } else {
        imgp = vg4_intern_get_img_from_imgcnt(&cvtype->img_act, &iattr);
        if ((VGI_CVKEY_RELEASED(cvdata->keys.k[CV_KEY_RETURN]) && !VGI_CVKEY_PRESSED(cvdata->keys.m_left))
            || (VGI_CVKEY_RELEASED(cvdata->keys.m_left) && !VGI_CVKEY_PRESSED(cvdata->keys.k[CV_KEY_RETURN]))
           ) {
          if (cvtype->is_cancel) {
            cvdata->keys.k[CV_KEY_ESCAPE] = CVKEY_IS_NEWPRESSED;
          } else {
            *selname = cvitem->name;
          }
        }
      }
    }

    if (imgp == NULL) {  /* default */
      imgp = vg4_intern_get_img_from_imgcnt(&cvtype->img_dfl, &iattr);
      if (cvitem->isdisabled) {  /* disabled */
        iattr.pixel.pixelcolor = VG_PIXELCOLOR_GREY;
        iattr.pixel.brightness = iattr.pixel.brightness * 4 / 5;
      } else {
        drawborder = VG_TRUE;
      }
    }

    /* draw out */
    if (imgp != NULL) {
      vg4->image->clear(cvitem->rimg);
      posb = canvas_set_posi(&cvitem->rect, imgp, cvtype->orientation, cvtype->v_orientation);
      vg4->image->copy(cvitem->rimg, imgp, &posb, &iattr);
      posb.pos = VG_POS_UPPER_LEFT;
      posb.x = cvdata->xadd + cvitem->rect.x;
      posb.y = cvdata->yadd + cvitem->rect.y;
      vg4->window->copy(cvitem->rimg, &posb, NULL);
      if (cvdata->hasfocus && drawborder) {
        canvas_draw_actborder(&cvitem->rect, cvdata->xadd, cvdata->yadd);
      }
    }
  }
} /* Ende t_draw */


static void
t_destroy(struct vgi_canvas_item *cvitem)
{
  struct cv_type *cvtype;

  if (cvitem == NULL) { return; }
  cvtype = (struct cv_type *)cvitem->vstrct;
  if (cvtype == NULL) { return; }

  vg4_intern_free_imgcnt(&cvtype->img_dfl);
  vg4_intern_free_imgcnt(&cvtype->img_act);
  vg4_intern_free_imgcnt(&cvtype->img_sel);

  free(cvtype);
} /* Ende t_destroy */


static void
t_dump(FILE *outfp, struct vgi_canvas_item *cvitem)
{
  struct cv_type *cvtype;
  const char *namptr;

  if (outfp == NULL || cvitem == NULL) { return; }
  cvtype = (struct cv_type *)cvitem->vstrct;
  if (cvtype == NULL) { return; }

  fprintf(outfp, "    type=button\n");
  fprintf(outfp, "    is-cancel=%s\n", (cvtype->is_cancel ? "yes" : "no"));
  namptr = vg4_intern_get_name_from_imgcnt(&cvtype->img_dfl);
  fprintf(outfp, "    img-dfl=%s\n", namptr);
  namptr = vg4_intern_get_name_from_imgcnt(&cvtype->img_act);
  fprintf(outfp, "    img-act=%s\n", namptr);
  namptr = vg4_intern_get_name_from_imgcnt(&cvtype->img_sel);
  fprintf(outfp, "    img-sel=%s\n", namptr);
} /* Ende t_dump */
