#ifndef SML3__NW_SUPPORT_H_
#define SML3__NW_SUPPORT_H_

/* ++++++++++++++++++++
 * +++ Beschreibung +++
 * ++++++++++++++++++++

  Unterstuetzung gewisser Netzwerkfunktionen:
  - SML3_getaddrinfo()  -> getaddrinfo()
  - SML3_getnameinfo()  -> getnameinfo()
  - SML3_freeaddrinfo() -> freeaddrinfo()
  - SML3_inet_pton()    -> inet_pton()
  - SML3_inet_ntop()    -> inet_ntop()

  Dasselbe fuer UNIXPath-Verbindungen:
  - SML3_getaddrinfo_unix()
  - SML3_getnameinfo_unix()
  - SML3_freeaddrinfo_unix()
+++ */


/* ++++++++++++++++++ */
/* +++ Funktionen +++ */
/* ++++++++++++++++++ */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#ifndef SML3_HAVE_GETADDRINFO
struct addrinfo {
  int ai_flags;               /* AI_PASSIVE, AI_CANONNAME */
  int ai_family;              /* PF_xxx */
  int ai_socktype;            /* SOCK_xxx */
  int ai_protocol;            /* IPPROTO_xxx for IPv4_ and IPv6_ */
  size_t ai_addrlen;          /* length of ai_addr */
  char *ai_canonname;         /* canonical name for host */
  struct sockaddr *ai_addr;   /* binary address */
  struct addrinfo *ai_next;   /* next structure in linked list */
};

#define AI_PASSIVE      1  /* socket is intended for bind()+listen() */
#define AI_CANONNAME    2  /* return canonical name */
#ifndef AI_CLONE
# define AI_CLONE  4
#endif

#define NI_NUMERICHOST  1  /* Don't try to look up hostname */
#define NI_NUMERICSERV  2  /* Don't convert port number to name */
#define NI_NOFQDN       4  /* Only return nodename portion */
#define NI_NAMEREQD     8  /* Don't return numeric addresses */
#define NI_DGRAM       16  /* Look up UDP service rather than TCP */
#endif /* SML3_HAVE_GETADDRINFO */

#ifndef NI_MAXHOST
# define NI_MAXHOST   256  /* max hostname returned */
#endif

#ifndef NI_MAXSERV
# define NI_MAXSERV    32  /* max service name returned */
#endif

#ifndef AF_LOCAL
# define AF_LOCAL  AF_UNIX
#endif

#ifndef AF_UNSPEC
# define AF_UNSPEC AF_INET
#endif


/** SML3_getaddrinfo: siehe getaddrinfo()
 * Args: wie bei getaddrinfo()
 * Rueckgabe: 0 = OK oder -1 = Fehler
 * SML3-errno-Wert: EINVAL = Fehler Uebergabeparameter
 *                  EAGAIN = Namensaufloesung temporaer fehlgeschlagen
 *                  ENXIO = Namensaufloesung fehlgeschlagen
 *                  ENOMEM = Allokationsfehler
 */
extern int SML3_getaddrinfo(const char *, const char *, const struct addrinfo *, struct addrinfo **);


/** SML3_freeaddrinfo: siehe freeaddrinfo()
 * Args: wie bei freeaddrinfo()
 */
extern void SML3_freeaddrinfo(struct addrinfo *);


/** SML3_getnameinfo: siehe getnameinfo()
 * Args: wie bei getnameinfo()
 * Rueckgabe: 0 = OK oder -1 = Fehler
 * SML3-errno-Wert: EINVAL = Fehler Uebergabeparameter
 *                  EAGAIN = Namensaufloesung temporaer fehlgeschlagen
 *                  ENXIO = Namensaufloesung fehlgeschlagen
 *                  ENOMEM = Allokationsfehler
 */
extern int SML3_getnameinfo(const struct sockaddr *, socklen_t, char *, size_t, char *, size_t, int);


/** SML3_inet_pton: siehe inet_pton()
 * Args: wie bei inet_pton()
 * Rueckgabe: 1 = OK oder 0 = keine gueltige Adresse uebergeben oder -1 = Fehler
 * SML3-errno-Wert: EAFNOSUPPORT = Adressfamilie nicht unterstuetzt
 */
extern int SML3_inet_pton(int, const char *, void *);


/** SML3_inet_ntop: siehe inet_ntop()
 * Args: wie bei inet_ntop()
 * Rueckgabe: Pointer auf 3.Arg oder NULL = Fehler
 * SML3-errno-Wert: EAFNOSUPPORT = Adressfamilie nicht unterstuetzt
 *                  ENOSPC = nicht genug Platz im 3.Arg
 */
extern const char * SML3_inet_ntop(int, const void *, char *, size_t);


/** SML3_getaddrinfo_unix: wie getaddrinfo() fuer UNIXPath-Verbindungen
 * 1.Arg: UNIX-Socketpfad
 * weitere Args: wie bei getaddrinfo() ab 3.Arg
 * Rueckgabe: 0 = OK oder -1 = Fehler
 * SML3-errno-Wert: EINVAL = Fehler Uebergabeparameter
 *                  ENOMEM = Allokationsfehler
 */
extern int SML3_getaddrinfo_unix(const char *, const struct addrinfo *, struct addrinfo **);


/** SML3_getnameinfo_unix: wie getnameinfo() fuer UNIXPath-Verbindungen
 * 1.Arg: wie bei getnameinfo() 1.Arg
 * 2.+3.Arg: wie bei getnameinfo() 3.+4.Arg fuer Rueckgabe UNIX-Socket-Pfad
 * Rueckgabe: 0 = OK oder -1 = Fehler
 * SML3-errno-Wert: EINVAL = Fehler Uebergabeparameter
 *                  ENOMEM = Allokationsfehler
 */
extern int SML3_getnameinfo_unix(const struct sockaddr *, char *, size_t);


/** SML3_freeaddrinfo_unix: wie freeaddrinfo() fuer UNIXPath-Verbindungen
 * Args: wie bei freeaddrinfo()
 */
extern void SML3_freeaddrinfo_unix(struct addrinfo *);

#endif /* SML3__NW_SUPPORT_H_ */
