/* sml3_latin.c: UTF-8 und Latein Konvertierungen */

/* Copyright 2012-2017 Kurt Nienhaus
 *
 * This file is part of libsammel3.
 * libsammel3 is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * libsammel3 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with libsammel3.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"
#ifdef SML3_HAVE_PTHREAD
# include <pthread.h>
#endif
#include "sml3_fehler.h"
#include "sml3_util.h"
#include "sml3_gummi.h"
#include "sml3_latin.h"


/* +++ ASCII-Tabellen +++ */

static const char *asc000[] = {
  "\x00", "\x01", "\x02", "\x03", "\x04", "\x05", "\x06", "\x07",  /* 00 */
  "\x08", "\x09", "\x0a", "\x0b", "\x0c", "\x0d", "\x0e", "\x0f",  /* 08 */
  "\x10", "\x11", "\x12", "\x13", "\x14", "\x15", "\x16", "\x17",  /* 10 */
  "\x18", "\x19", "\x1a", "\x1b", "\x1c", "\x1d", "\x1e", "\x1f",  /* 18 */
  "\x20", "\x21", "\x22", "\x23", "\x24", "\x25", "\x26", "\x27",  /* 20 */
  "\x28", "\x29", "\x2a", "\x2b", "\x2c", "\x2d", "\x2e", "\x2f",  /* 28 */
  "\x30", "\x31", "\x32", "\x33", "\x34", "\x35", "\x36", "\x37",  /* 30 */
  "\x38", "\x39", "\x3a", "\x3b", "\x3c", "\x3d", "\x3e", "\x3f",  /* 38 */
  "\x40", "\x41", "\x42", "\x43", "\x44", "\x45", "\x46", "\x47",  /* 40 */
  "\x48", "\x49", "\x4a", "\x4b", "\x4c", "\x4d", "\x4e", "\x4f",  /* 48 */
  "\x50", "\x51", "\x52", "\x53", "\x54", "\x55", "\x56", "\x57",  /* 50 */
  "\x58", "\x59", "\x5a", "\x5b", "\x5c", "\x5d", "\x5e", "\x5f",  /* 58 */
  "\x60", "\x61", "\x62", "\x63", "\x64", "\x65", "\x66", "\x67",  /* 60 */
  "\x68", "\x69", "\x6a", "\x6b", "\x6c", "\x6d", "\x6e", "\x6f",  /* 68 */
  "\x70", "\x71", "\x72", "\x73", "\x74", "\x75", "\x76", "\x77",  /* 70 */
  "\x78", "\x79", "\x7a", "\x7b", "\x7c", "\x7d", "\x7e", "\x7f",  /* 78 */
  "EUR", NULL, "'", "f", "\"", "...", "+", "++",                   /* 80 - enh. */
  "^", "o/oo", "S", "<", "OE", NULL, NULL, NULL,                   /* 88 - enh. */
  NULL, "'", "'", "\"", "\"", ".", "-", "-",                       /* 90 - enh. */
  "~", "TM", "s", ">", "oe", NULL, NULL, "Y",                      /* 98 - enh. */
  " ", "!", "Cent", "Pfund", NULL, "Yen", "|", "Paragraph",        /* a0 */
  NULL, "(C)", "a", "\"", NULL, NULL, "(R)", NULL,                 /* a8 */
  "Grad", "+-", "2", "3", "'", "u", NULL, ".",                     /* b0 */
  NULL, "1", "o", "\"", "1/4", "1/2", "3/4", "?",                  /* b8 */
  "A", "A", "A", "A", "A?", "A", "AE", "C",                        /* c0 */
  "E", "E", "E", "E", "I", "I", "I", "I",                          /* c8 */
  "D", "N", "O", "O", "O", "O", "O?", "x",                         /* d0 */
  "O", "U", "U", "U", "U?", "Y", "P", "ss",                        /* d8 */
  "a", "a", "a", "a", "a?", "a", "ae", "c",                        /* e0 */
  "e", "e", "e", "e", "i", "i", "i", "i",                          /* e8 */
  "d", "n", "o", "o", "o", "o", "o?", "/",                         /* f0 */
  "o", "u", "u", "u", "u?", "y", "p", "y"                          /* f8 */
};

static const char *asc001[] = {
  "A", "a", "A", "a", "A", "a", "C", "c",                          /* 00 */
  "C", "c", "C", "c", "C", "c", "D", "d",                          /* 08 */
  "D", "d", "E", "e", "E", "e", "E", "e",                          /* 10 */
  "E", "e", "E", "e", "G", "g", "G", "g",                          /* 18 */
  "G", "g", "G", "g", "H", "h", "H", "h",                          /* 20 */
  "I", "i", "I", "i", "I", "i", "I", "i",                          /* 28 */
  "I", "i", "IJ", "ij", "J", "j", "K", "k",                        /* 30 */
  "k", "L", "l", "L", "l", "L", "l", "L",                          /* 38 */
  "l", "L", "l", "N", "n", "N", "n", "N",                          /* 40 */
  "n", "'n", "NG", "ng", "O", "o", "O", "o",                       /* 48 */
  "O", "o", "OE", "oe", "R", "r", "R", "r",                        /* 50 */
  "R", "r", "S", "s", "S", "s", "S", "s",                          /* 58 */
  "S", "s", "T", "t", "T", "t", "T", "t",                          /* 60 */
  "U", "u", "U", "u", "U", "u", "U", "u",                          /* 68 */
  "U", "u", "U", "u", "W", "w", "Y", "y",                          /* 70 */
  "y", "Z", "z", "Z", "z", "Z", "z", "s",                          /* 78 */
  "b", "B", "B", "b", NULL, NULL, NULL, NULL,                      /* 80 */
  NULL, "D", "D", "D", "d", NULL, "E", "e",                        /* 88 */
  "E", "F", "f", "G", NULL, "hv", "I", "I",                        /* 90 */
  "K", "k", "l", NULL, "M", "N", "n", "O",                         /* 98 */
  "O", "o", "OI", "oi", "P", "p", "YR", "S",                       /* a0 */
  "s", NULL, NULL, "t", "T", "t", "T", "U",                        /* a8 */
  "u", "Y", "V", "Y", "y", "Z", "z", "Z",                          /* b0 */
  "Z", "z", "z", "2", NULL, NULL, NULL, NULL,                      /* b8 */
  NULL, NULL, NULL, "!", "DZ", "Dz", "dz", "LJ",                   /* c0 */
  "Lj", "lj", "NJ", "Nj", "nj", "A", "a", "I",                     /* c8 */
  "i", "O", "o", "U", "u", "U?", "U?", "U?",                       /* d0 */
  "u?", "U?", "u?", "U?", "u?", "e", "A?", "a?",                   /* d8 */
  "A", "a", "AE", "ae", "G", "g", "G", "g",                        /* e0 */
  "K", "k", "O", "o", "O", "o", "Z", "z",                          /* e8 */
  "j", "DZ", "Dz", "dz", "G", "g", NULL, NULL,                     /* f0 */
  "N", "n", "A", "a", "AE", "ae", "O", "o"                         /* f8 */
};

static const char *asc002[] = {
  "A", "a", "A", "a", "E", "e", "E", "e",                          /* 00 */
  "I", "i", "I", "i", "O", "o", "O", "o",                          /* 08 */
  "R", "r", "R", "r", "U", "u", "U", "u",                          /* 10 */
  "S", "s", "T", "t", "Z", "z", "H", "h",                          /* 18 */
  "N", "d", "OU", "ou", "Z", "z", "A", "a",                        /* 20 */
  "E", "e", "O?", "o?", "O", "o", "O", "o",                        /* 28 */
  "O", "o", "Y", "y", "l", "n", "t", "j",                          /* 30 */
  "db", "qp", "A", "C", "c", "L", "T", "s",                        /* 38 */
  "z", NULL, NULL, "B", "U", "V", "E", "e",                        /* 40 */
  "J", "j", "Q", "q", "R", "r", "Y", "y",                          /* 48 */
  "a", "a", "a", "b", NULL, "c", "d", "d",                         /* 50 */
  "e", "e", "e", "e", "e", "e", "e", "j",                          /* 58 */
  "g", "g", "G", "g", NULL, "h", "h", "h",                         /* 60 */
  "i", "i", "I", "l", "l", "l", "lz", "m",                         /* 68 */
  "m", "m", "n", "n", "N", "o", "OE", "o",                         /* 70 */
  NULL, "r", "r", "r", "r", "r", "r", "r",                         /* 78 */
  "R", "R", "s", "s", "j", "s", "s", "t",                          /* 80 */
  "t", "u", "y", "v", "v", "w", "y", "Y",                          /* 88 */
  "z", "z", "z", "z", NULL, NULL, NULL, "C",                       /* 90 */
  NULL, "B", "e", "G", "H", "j", "k", "L",                         /* 98 */
  "q", NULL, NULL, "dz", "dz", "dz", "ts", "ts",                   /* a0 */
  "tc", "fng", "ls", "lz", NULL, NULL, "h", "h",                   /* a8 */
  "h", "h", "j", "r", "r", "r", "R", "w",                          /* b0 */
  "y", "'", "\"", "'", "'", "'", "(", ")",                         /* b8 */
  NULL, NULL, "<", ">", "^", NULL, "^", NULL,                      /* c0 */
  "'", "-", "'", "'", ",", "_", ",", ",",                          /* c8 */
  ":", ".", ")", "(", NULL, NULL, "+", "-",                        /* d0 */
  NULL, NULL, "Grad", NULL, "~", "\"", NULL, NULL,                 /* d8 */
  NULL, "l", "s", "x", NULL, NULL, NULL, NULL,                     /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\"", NULL,                  /* e8 */
  "^", "<", ">", ".", "'", "\"", "\"", "~",                        /* f0 */
  ":", NULL, NULL, NULL, NULL, NULL, NULL, NULL                    /* f8 */
};

static const char *asc01d[] = {
  "A", "AE", "ae", "B", "C", "D", "D", "E",                        /* 00 */
  "e", "i", "J", "K", "L", "M", "N", "O",                          /* 08 */
  NULL, "o", "o", "o", "oe", "OU", NULL, NULL,                     /* 10 */
  "P", "R", "R", "T", "U", "u", "u?", "m",                         /* 18 */
  "V", "W", "Z", "Z", NULL, NULL, NULL, NULL,                      /* 20 */
  NULL, "P", NULL, NULL, "A", "AE", "B", "B",                      /* 28 */
  "D", "E", "E", "G", "H", "I", "J", "K",                          /* 30 */
  "L", "M", "N", "N", "O", "OU", "P", "R",                         /* 38 */
  "T", "U", "W", "a", "a", "a", "ae", "b",                         /* 40 */
  "d", "e", "e", "e", "e", "g", "i", "k",                          /* 48 */
  "m", "ng", "o", NULL, NULL, NULL, "p", "t",                      /* 50 */
  "u", "u", "m", "v", NULL, "b", NULL, "d",                        /* 58 */
  NULL, "x", "i", "r", "u", "v", "b", NULL,                        /* 60 */
  "p", NULL, "x", "ue", "b", "d", "f", "m",                        /* 68 */
  "n", "p", "r", "r", "s", "t", "z", "g",                          /* 70 */
  NULL, "g", "th", "I", "i", "p", "U", "y",                        /* 78 */
  "b", "d", "f", "g", "k", "l", "m", "n",                          /* 80 */
  "p", "r", "s", "s", "v", "x", "z", "a",                          /* 88 */
  "a", "d", "e", "e", "e", "e", "i", NULL,                         /* 90 */
  "s", "u", "z", "a", "c", "c", "d", "e",                          /* 98 */
  "f", "j", "g", "h", "i", "i", "I", "I",                          /* a0 */
  "j", "l", "l", "L", "m", "m", "n", "n",                          /* a8 */
  "N", "o", NULL, "s", "s", "t", "u", "y",                         /* b0 */
  "U", "v", "v", "z", "z", "z", "z", NULL,                         /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, "a", "ae", "ao", "av", "c",                    /* d0 */
  "d", "d", "g", "G", "k", "l", "L", "M",                          /* d8 */
  "n", "N", "R", "r", "s", "s", "z", NULL,                         /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asc01e[] = {
  "A", "a", "B", "b", "B", "b", "B", "b",                          /* 00 */
  "C", "c", "D", "d", "D", "d", "D", "d",                          /* 08 */
  "D", "d", "D", "d", "E", "e", "E", "e",                          /* 10 */
  "E", "e", "E", "e", "E", "e", "F", "f",                          /* 18 */
  "G", "g", "H", "h", "H", "h", "H", "h",                          /* 20 */
  "H", "h", "H", "h", "I", "i", "I", "i",                          /* 28 */
  "K", "k", "K", "k", "K", "k", "L", "l",                          /* 30 */
  "L", "l", "L", "l", "L", "l", "M", "m",                          /* 38 */
  "M", "m", "M", "m", "N", "n", "N", "n",                          /* 40 */
  "N", "n", "N", "n", "O", "o", "O?", "o?",                        /* 48 */
  "O", "o", "O", "o", "P", "p", "P", "p",                          /* 50 */
  "R", "r", "R", "r", "R", "r", "R", "r",                          /* 58 */
  "S", "s", "S", "s", "S", "s", "S", "s",                          /* 60 */
  "S", "s", "T", "t", "T", "t", "T", "t",                          /* 68 */
  "T", "t", "U", "u", "U", "u", "U", "u",                          /* 70 */
  "U", "u", "U?", "u?", "V", "v", "V", "v",                        /* 78 */
  "W", "w", "W", "w", "W", "w", "W", "w",                          /* 80 */
  "W", "w", "X", "x", "X", "x", "Y", "y",                          /* 88 */
  "Z", "z", "Z", "z", "Z", "z", "h", "t",                          /* 90 */
  "w", "y", "a", "s", "s", "s", "SS", "d",                         /* 98 */
  "A", "a", "A", "a", "A", "a", "A", "a",                          /* a0 */
  "A", "a", "A", "a", "A", "a", "A", "a",                          /* a8 */
  "A", "a", "A", "a", "A", "a", "A", "a",                          /* b0 */
  "E", "e", "E", "e", "E", "e", "E", "e",                          /* b8 */
  "E", "e", "E", "e", "E", "e", "E", "e",                          /* c0 */
  "I", "i", "I", "i", "O", "o", "O", "o",                          /* c8 */
  "O", "o", "O", "o", "O", "o", "O", "o",                          /* d0 */
  "O", "o", "O", "o", "O", "o", "O", "o",                          /* d8 */
  "O", "o", "O", "o", "U", "u", "U", "u",                          /* e0 */
  "U", "u", "U", "u", "U", "u", "U", "u",                          /* e8 */
  "U", "u", "Y", "y", "Y", "y", "Y", "y",                          /* f0 */
  "Y", "y", "LL", "ll", "V", "v", "Y", "y"                         /* f8 */
};

static const char *asc020[] = {
  " ", " ", " ", " ", " ", " ", " ", " ",                          /* 00 */
  " ", " ", " ", " ", " ", " ", " ", " ",                          /* 08 */
  "-", "-", "-", "-", "-", "-", "||", "_",                         /* 10 */
  "'", "'", "'", "'", "\"", "\"", "\"", "\"",                      /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, " ",                   /* 28 */
  "o/oo", "o/ooo", NULL, NULL, NULL, NULL, NULL, NULL,             /* 30 */
  "^", "'", "'", NULL, "!!", NULL, NULL, NULL,                     /* 38 */
  NULL, "^", "*", "-", "/", "[", "]", "??",                        /* 40 */
  "?!", "!?", NULL, NULL, NULL, NULL, "*", ";",                    /* 48 */
  NULL, "*", "%", NULL, NULL, ".", NULL, NULL,                     /* 50 */
  NULL, NULL, ":", NULL, NULL, "|", "|", " ",                      /* 58 */
  NULL, NULL, NULL, " ", "+", NULL, NULL, NULL,                    /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  "0", "i", NULL, NULL, "4", "5", "6", "7",                        /* 70 */
  "8", "9", "+", "-", "=", "(", ")", "n",                          /* 78 */
  "0", "1", "2", "3", "4", "5", "6", "7",                          /* 80 */
  "8", "9", "+", "-", "=", "(", ")", NULL,                         /* 88 */
  "a", "e", "o", "x", "e", "h", "k", "l",                          /* 90 */
  "m", "n", "p", "s", "t", NULL, NULL, NULL,                       /* 98 */
  "EUR", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                 /* a0 */
  NULL, NULL, NULL, NULL, "EUR", NULL, NULL, NULL,                 /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, "|", "|", NULL, NULL, NULL, NULL,                    /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, "\\", "||", NULL,                  /* e0 */
  NULL, NULL, NULL, "//", NULL, NULL, NULL, NULL,                  /* e8 */
  "*", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                   /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asc021[] = {
  "a/c", "a/s", "C", "Grad C", NULL, "c/o", "c/u", "E",            /* 00 */
  NULL, "Grad F", "g", "H", "H", "H", "h", "h",                    /* 08 */
  "I", "I", "L", "l", "lb", "N", "No", "(P)",                      /* 10 */
  "P", "P", "Q", "R", "R", "R", "Px", "R",                         /* 18 */
  "SM", "TEL", "TM", "V", "Z", NULL, NULL, NULL,                   /* 20 */
  "Z", "i", "K", "A", "B", "C", "e", "e",                          /* 28 */
  "E", "F", "F", "M", "o", NULL, NULL, NULL,                       /* 30 */
  NULL, "i", "Q", "FAX", "PI", NULL, NULL, "PI",                   /* 38 */
  NULL, "G", "L", "L", "Y", "D", "d", "e",                         /* 40 */
  "i", "j", NULL, "&", NULL, "A/S", "F", NULL,                     /* 48 */
  "1/7", "1/9", "1/10", "1/3", "2/3", "1/5", "2/5", "3/5",         /* 50 */
  "4/5", "1/6", "5/6", "1/8", "3/8", "5/8", "7/8", NULL,           /* 58 */
  "I", "II", "III", "IV", "V", "VI", "VII", "VIII",                /* 60 */
  "IX", "X", "XI", "XII", "L", "C", "D", "M",                      /* 68 */
  "i", "ii", "iii", "iv", "v", "vi", "vii", "viii",                /* 70 */
  "ix", "x", "xi", "xii", "l", "c", "d", "m",                      /* 78 */
  "CD", NULL, NULL, "C", "c", NULL, NULL, NULL,                    /* 80 */
  NULL, "0/3", NULL, NULL, NULL, NULL, NULL, NULL,                 /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asc022[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, "-", "-+", "+", "/", "\\", "*",                      /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, ":=", "=:", NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, "<=", ">=", "<=", ">=",                  /* 60 */
  "<", ">", "<<", ">>", NULL, NULL, NULL, NULL,                    /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, "<", ">",                    /* d0 */
  "<<<", ">>>", NULL, NULL, "<=", ">=", NULL, NULL,                /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asc023[] = {
  NULL, NULL, NULL, "^", NULL, NULL, NULL, NULL,                   /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, "#",                   /* 10 */
  "x", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                   /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, "<", ">", NULL, NULL, NULL, NULL, NULL,                    /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  "~", ">", ",", NULL, NULL, NULL, ";", NULL,                      /* 68 */
  NULL, NULL, NULL, "i", "p", NULL, "a", "e",                      /* 70 */
  "i", NULL, "a", NULL, NULL, NULL, NULL, NULL,                    /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, "(", "(", "(", ")", ")",                       /* 98 */
  ")", "[", "[", "[", "]", "]", "]", "{",                          /* a0 */
  "{", "{", "|", "}", "}", "}", "|", "-",                          /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  "|", "|", "-", "-", "-", "-", NULL, NULL,                        /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  "|", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                   /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asc025[] = {
  "-", "-", "|", "|", "-", "-", "|", "|",                          /* 00 */
  "-", "-", "|", "|", NULL, NULL, NULL, NULL,                      /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, "-", "-", "|", "|",                      /* 48 */
  "=", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                   /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, "/", "\\", NULL, "-", NULL, "-", NULL,                     /* 70 */
  "-", NULL, "-", NULL, NULL, NULL, NULL, NULL,                    /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, ".", NULL,                   /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asc02c[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  "L", "l", "L", "P", "R", "a", "t", "H",                          /* 60 */
  "h", "K", "k", "Z", "z", "A", "M", "A",                          /* 68 */
  "A", "v", "W", "w", "v", "H", "h", NULL,                         /* 70 */
  "e", "r", "o", "E", "j", "V", "S", "Z",                          /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asc0a7[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, "H", "h",                    /* 20 */
  "TZ", "tz", "E", "e", NULL, NULL, NULL, NULL,                    /* 28 */
  "F", "S", "AA", "aa", "AO", "ao", "AU", "au",                    /* 30 */
  "AV", "av", "AV", "av", "AY", "ay", "C", "c",                    /* 38 */
  "K", "k", "K", "k", "K", "k", "L", "l",                          /* 40 */
  "L", "l", "O", "o", "O", "o", "OO", "oo",                        /* 48 */
  "P", "p", "P", "p", "P", "p", "Q", "q",                          /* 50 */
  "Q", "q", "R", "r", "R", "r", "V", "v",                          /* 58 */
  "VY", "vy", "Z", "z", "P", "p", "P", "p",                        /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, "D", "d", "F", "f", "G", "G", "g",                         /* 78 */
  "L", "l", "R", "r", "S", "s", "T", "t",                          /* 80 */
  "~", ":", "=", NULL, NULL, "h", "l", NULL,                       /* 88 */
  "N", "n", NULL, NULL, NULL, NULL, NULL, NULL,                    /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  "G", "g", "K", "k", "N", "n", "R", "r",                          /* a0 */
  "S", "s", NULL, NULL, NULL, NULL, NULL, NULL,                    /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, "M", "F", "P", "M", "I", "M"                         /* f8 */
};

static const char *asc0fb[] = {
  "ff", "fi", "fl", "ffi", "ffl", "st", "st", NULL,                /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asc0ff[] = {
  " ", "!", "\"", "#", "$", "%", "&", "'",                         /* 00 */
  "(", ")", "*", "+", ",", "-", ".", "/",                          /* 08 */
  "0", "1", "2", "3", "4", "5", "6", "7",                          /* 10 */
  "8", "9", ":", ";", "<", "=", ">", "?",                          /* 18 */
  "@", "A", "B", "C", "D", "E", "F", "G",                          /* 20 */
  "H", "I", "J", "K", "L", "M", "N", "O",                          /* 28 */
  "P", "Q", "R", "S", "T", "U", "V", "W",                          /* 30 */
  "X", "Y", "Z", "[", "\\", "]", "^", "_",                         /* 38 */
  "`", "a", "b", "c", "d", "e", "f", "g",                          /* 40 */
  "h", "i", "j", "k", "l", "m", "n", "o",                          /* 48 */
  "p", "q", "r", "s", "t", "u", "v", "w",                          /* 50 */
  "x", "y", "z", "{", "|", "}", "~", "(",                          /* 58 */
  ")", ".", NULL, NULL, ",", ".", NULL, NULL,                      /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  "Cent", "Pfund", NULL, NULL, "|", "Yen", NULL, NULL,             /* e0 */
  "|", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                   /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asc1d4[] = {
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* 00 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* 08 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* 10 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* 18 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* 20 */
  "o", "p", "q", "r", "s", "t", "u", "v",                          /* 28 */
  "w", "x", "y", "z", "A", "B", "C", "D",                          /* 30 */
  "E", "F", "G", "H", "I", "J", "K", "L",                          /* 38 */
  "M", "N", "O", "P", "Q", "R", "S", "T",                          /* 40 */
  "U", "V", "W", "X", "Y", "Z", "a", "b",                          /* 48 */
  "c", "d", "e", "f", "g", "h", "i", "j",                          /* 50 */
  "k", "l", "m", "n", "o", "p", "q", "r",                          /* 58 */
  "s", "t", "u", "v", "w", "x", "y", "z",                          /* 60 */
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* 68 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* 70 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* 78 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* 80 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* 88 */
  "o", "p", "q", "r", "s", "t", "u", "v",                          /* 90 */
  "w", "x", "y", "z", "A", "B", "C", "D",                          /* 98 */
  "E", "F", "G", "H", "I", "J", "K", "L",                          /* a0 */
  "M", "N", "O", "P", "Q", "R", "S", "T",                          /* a8 */
  "U", "V", "W", "X", "Y", "Z", "a", "b",                          /* b0 */
  "c", "d", "e", "f", "g", "h", "i", "j",                          /* b8 */
  "k", "l", "m", "n", "o", "p", "q", "r",                          /* c0 */
  "s", "t", "u", "v", "w", "x", "y", "z",                          /* c8 */
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* d0 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* d8 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* e0 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* e8 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* f0 */
  "o", "p", "q", "r", "s", "t", "u", "v"                           /* f8 */
};

static const char *asc1d5[] = {
  "w", "x", "y", "z", "A", "B", "C", "D",                          /* 00 */
  "E", "F", "G", "H", "I", "J", "K", "L",                          /* 08 */
  "M", "N", "O", "P", "Q", "R", "S", "T",                          /* 10 */
  "U", "V", "W", "X", "Y", "Z", "a", "b",                          /* 18 */
  "c", "d", "e", "f", "g", "h", "i", "j",                          /* 20 */
  "k", "l", "m", "n", "o", "p", "q", "r",                          /* 28 */
  "s", "t", "u", "v", "w", "x", "y", "z",                          /* 30 */
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* 38 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* 40 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* 48 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* 50 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* 58 */
  "o", "p", "q", "r", "s", "t", "u", "v",                          /* 60 */
  "w", "x", "y", "z", "A", "B", "C", "D",                          /* 68 */
  "E", "F", "G", "H", "I", "J", "K", "L",                          /* 70 */
  "M", "N", "O", "P", "Q", "R", "S", "T",                          /* 78 */
  "U", "V", "W", "X", "Y", "Z", "a", "b",                          /* 80 */
  "c", "d", "e", "f", "g", "h", "i", "j",                          /* 88 */
  "k", "l", "m", "n", "o", "p", "q", "r",                          /* 90 */
  "s", "t", "u", "v", "w", "x", "y", "z",                          /* 98 */
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* a0 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* a8 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* b0 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* b8 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* c0 */
  "o", "p", "q", "r", "s", "t", "u", "v",                          /* c8 */
  "w", "x", "y", "z", "A", "B", "C", "D",                          /* d0 */
  "E", "F", "G", "H", "I", "J", "K", "L",                          /* d8 */
  "M", "N", "O", "P", "Q", "R", "S", "T",                          /* e0 */
  "U", "V", "W", "X", "Y", "Z", "a", "b",                          /* e8 */
  "c", "d", "e", "f", "g", "h", "i", "j",                          /* f0 */
  "k", "l", "m", "n", "o", "p", "q", "r"                           /* f8 */
};

static const char *asc1d6[] = {
  "s", "t", "u", "v", "w", "x", "y", "z",                          /* 00 */
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* 08 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* 10 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* 18 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* 20 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* 28 */
  "o", "p", "q", "r", "s", "t", "u", "v",                          /* 30 */
  "w", "x", "y", "z", "A", "B", "C", "D",                          /* 38 */
  "E", "F", "G", "H", "I", "J", "K", "L",                          /* 40 */
  "M", "N", "O", "P", "Q", "R", "S", "T",                          /* 48 */
  "U", "V", "W", "X", "Y", "Z", "a", "b",                          /* 50 */
  "c", "d", "e", "f", "g", "h", "i", "j",                          /* 58 */
  "k", "l", "m", "n", "o", "p", "q", "r",                          /* 60 */
  "s", "t", "u", "v", "w", "x", "y", "z",                          /* 68 */
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* 70 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* 78 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* 80 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* 88 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* 90 */
  "o", "p", "q", "r", "s", "t", "u", "v",                          /* 98 */
  "w", "x", "y", "z", "i", "j", NULL, NULL,                        /* a0 */
  "A", "B", NULL, NULL, "E", "Z", "H", NULL,                       /* a8 */
  "I", "K", NULL, "M", "N", NULL, "O", "PI",                       /* b0 */
  "P", NULL, NULL, "T", "Y", NULL, "X", NULL,                      /* b8 */
  NULL, NULL, "a", "b", NULL, "d", "e", "z",                       /* c0 */
  NULL, NULL, "i", "k", NULL, "u", "v", NULL,                      /* c8 */
  "o", "pi", "p", NULL, NULL, "t", "u", NULL,                      /* d0 */
  "x", NULL, NULL, NULL, "e", NULL, "k", NULL,                     /* d8 */
  "p", NULL, "A", "B", NULL, NULL, "E", "Z",                       /* e0 */
  "H", NULL, "I", "K", NULL, "M", "N", NULL,                       /* e8 */
  "O", "PI", "P", NULL, NULL, "T", "Y", NULL,                      /* f0 */
  "X", NULL, NULL, NULL, "a", "b", NULL, "d"                       /* f8 */
};

static const char *asc1d7[] = {
  "e", "z", NULL, NULL, "i", "k", NULL, "u",                       /* 00 */
  "v", NULL, "o", "PI", "p", NULL, NULL, "t",                      /* 08 */
  "u", NULL, "x", NULL, NULL, NULL, "e", NULL,                     /* 10 */
  "k", NULL, "p", NULL, "A", "B", NULL, NULL,                      /* 18 */
  "E", "Z", "H", NULL, "I", "K", NULL, "M",                        /* 20 */
  "N", NULL, "O", "PI", "P", NULL, NULL, "T",                      /* 28 */
  "Y", "X", NULL, NULL, NULL, NULL, "a", "b",                      /* 30 */
  NULL, "d", "e", "z", NULL, NULL, "i", "k",                       /* 38 */
  NULL, "u", "v", NULL, "o", "PI", "p", NULL,                      /* 40 */
  NULL, "t", "u", NULL, "x", NULL, NULL, NULL,                     /* 48 */
  "e", NULL, "k", NULL, "p", NULL, "A", "B",                       /* 50 */
  NULL, NULL, "E", "Z", "H", NULL, "I", "K",                       /* 58 */
  NULL, "M", "N", NULL, "O", "PI", "P", NULL,                      /* 60 */
  NULL, "T", "Y", NULL, "X", NULL, NULL, NULL,                     /* 68 */
  "a", "b", NULL, "d", "e", "z", NULL, NULL,                       /* 70 */
  "i", "k", NULL, "u", "v", NULL, "o", "PI",                       /* 78 */
  "p", NULL, NULL, "t", "u", NULL, "x", NULL,                      /* 80 */
  NULL, NULL, "e", NULL, "k", NULL, "p", NULL,                     /* 88 */
  "A", "B", NULL, NULL, "E", "Z", "H", NULL,                       /* 90 */
  "I", "K", NULL, "M", "N", NULL, "O", "PI",                       /* 98 */
  "P", NULL, NULL, "T", "Y", NULL, "X", NULL,                      /* a0 */
  NULL, NULL, "a", "b", NULL, "d", "e", "z",                       /* a8 */
  NULL, NULL, "i", "k", NULL, "u", "v", NULL,                      /* b0 */
  "o", "PI", "p", NULL, NULL, "t", "u", NULL,                      /* b8 */
  "x", NULL, NULL, NULL, "e", NULL, "k", NULL,                     /* c0 */
  "p", NULL, "F", "f", NULL, NULL, "0", "1",                       /* c8 */
  "2", "3", "4", "5", "6", "7", "8", "9",                          /* d0 */
  "0", "1", "2", "3", "4", "5", "6", "7",                          /* d8 */
  "8", "9", "0", "1", "2", "3", "4", "5",                          /* e0 */
  "6", "7", "8", "9", "0", "1", "2", "3",                          /* e8 */
  "4", "5", "6", "7", "8", "9", "0", "1",                          /* f0 */
  "2", "3", "4", "5", "6", "7", "8", "9"                           /* f8 */
};

static const char *asce00[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  " ", "!", "\"", "#", "$", "%", "&", "'",                         /* 20 */
  "(", ")", "*", "+", ",", "-", ".", "/",                          /* 28 */
  "0", "1", "2", "3", "4", "5", "6", "7",                          /* 30 */
  "8", "9", ":", ";", "<", "=", ">", "?",                          /* 38 */
  "@", "A", "B", "C", "D", "E", "F", "G",                          /* 40 */
  "H", "I", "J", "K", "L", "M", "N", "O",                          /* 48 */
  "P", "Q", "R", "S", "T", "U", "V", "W",                          /* 50 */
  "X", "Y", "Z", "[", "\\", "]", "^", "_",                         /* 58 */
  "`", "a", "b", "c", "d", "e", "f", "g",                          /* 60 */
  "h", "i", "j", "k", "l", "m", "n", "o",                          /* 68 */
  "p", "q", "r", "s", "t", "u", "v", "w",                          /* 70 */
  "x", "y", "z", "{", "|", "}", "~", NULL,                         /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};


/* +++ ISO-8859-1-Supplement-Tabellen +++ */

static const char *iso000[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  "\x80", "\x81", "\x82", "\x83", "\x84", "\x85", "\x86", "\x87",  /* 80 - enh. */
  "\x88", "\x89", "\x8a", "\x8b", "\x8c", "\x8d", "\x8e", "\x8f",  /* 88 - enh. */
  "\x90", "\x91", "\x92", "\x93", "\x94", "\x95", "\x96", "\x97",  /* 90 - enh. */
  "\x98", "\x99", "\x9a", "\x9b", "\x9c", "\x9d", "\x9e", "\x9f",  /* 98 - enh. */
  " ", "\xa1", "\xa2", "\xa3", "\xa4", "\xa5", "\xa6", "\xa7",     /* a0 */
  "\xa8", "\xa9", "\xaa", "\xab", "\xac", "\xad", "\xae", "\xaf",  /* a8 */
  "\xb0", "\xb1", "\xb2", "\xb3", "\xb4", "\xb5", "\xb6", "\xb7",  /* b0 */
  "\xb8", "\xb9", "\xba", "\xbb", "\xbc", "\xbd", "\xbe", "\xbf",  /* b8 */
  "\xc0", "\xc1", "\xc2", "\xc3", "\xc4", "\xc5", "\xc6", "\xc7",  /* c0 */
  "\xc8", "\xc9", "\xca", "\xcb", "\xcc", "\xcd", "\xce", "\xcf",  /* c8 */
  "\xd0", "\xd1", "\xd2", "\xd3", "\xd4", "\xd5", "\xd6", "\xd7",  /* d0 */
  "\xd8", "\xd9", "\xda", "\xdb", "\xdc", "\xdd", "\xde", "\xdf",  /* d8 */
  "\xe0", "\xe1", "\xe2", "\xe3", "\xe4", "\xe5", "\xe6", "\xe7",  /* e0 */
  "\xe8", "\xe9", "\xea", "\xeb", "\xec", "\xed", "\xee", "\xef",  /* e8 */
  "\xf0", "\xf1", "\xf2", "\xf3", "\xf4", "\xf5", "\xf6", "\xf7",  /* f0 */
  "\xf8", "\xf9", "\xfa", "\xfb", "\xfc", "\xfd", "\xfe", "\xff"   /* f8 */
};

static const char *iso001[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  "\xd0", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, "\xd0", NULL, NULL, NULL, NULL, NULL, NULL,                /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, "\xdc", "\xfc", "\xdc",            /* d0 */
  "\xfc", "\xdc", "\xfc", "\xdc", "\xfc", NULL, "\xc4", "\xe4",    /* d8 */
  NULL, NULL, "\xc6", "\xe6", NULL, NULL, NULL, NULL,              /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, "\xc5", "\xe5", "\xc6", "\xe6", "\xd8", "\xf8"       /* f8 */
};

static const char *iso002[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, "\xd6", "\xf6", "\xd5", "\xf5", NULL, NULL,          /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, "\xb0", NULL, NULL, NULL, NULL, NULL,                /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *iso01d[] = {
  NULL, "\xc6", "\xe6", NULL, NULL, NULL, "\xd0", NULL,            /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, "\xf8", NULL, NULL, NULL, NULL,                /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xfc", NULL,                /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, "\xc6", NULL, NULL,                /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, "\xe6", NULL, NULL,                /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, "\xdf", NULL, NULL,                /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xdf", NULL,                /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xf0", NULL,                /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, "\xe6", NULL, NULL, "\xe7",              /* d0 */
  NULL, "\xf0", NULL, NULL, NULL, NULL, NULL, NULL,                /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *iso01e[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  "\xc7", "\xe7", NULL, NULL, NULL, NULL, NULL, NULL,              /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xd6", "\xf6",              /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, "\xdc", "\xfc", NULL, NULL, NULL, NULL, NULL,              /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xdf", NULL,                /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, "\xc2", "\xe2", NULL, NULL,              /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xca", "\xea",              /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  "\xd4", "\xf4", "\xd3", "\xf3", "\xd2", NULL, "\xf2", NULL,      /* d8 */
  "\xd5", "\xf5", NULL, NULL, NULL, NULL, NULL, NULL,              /* e0 */
  "\xda", "\xfa", "\xd9", "\xf9", NULL, NULL, NULL, NULL,          /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *iso020[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, "\xb7",                /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xaf", NULL,                /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, "\xb6", NULL, NULL, NULL, NULL,                /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, "\xa2", NULL, NULL,                /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *iso021[] = {
  NULL, NULL, NULL, "\xb0""C", NULL, NULL, NULL, NULL,             /* 00 */
  NULL, "\xb0""F", NULL, NULL, NULL, NULL, NULL, NULL,             /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, "\xc5", NULL, NULL, NULL, NULL,                /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *iso022[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, "\xb1", NULL, NULL, NULL, NULL,                /* 10 */
  "\xb0", "\xb7", NULL, NULL, NULL, NULL, NULL, NULL,              /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, "\xb7", NULL, NULL,                /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *iso023[] = {
  "\xf8", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  "\xac", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                /* 10 */
  "\xd7", "\xac", NULL, NULL, NULL, NULL, NULL, NULL,              /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, "\xba", NULL, NULL, NULL, NULL,                /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, "\xa2",                /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *iso025[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, "\xa6", "\xa6",              /* 00 */
  NULL, NULL, "\xa6", "\xa6", NULL, NULL, NULL, NULL,              /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xa6", "\xa6",              /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xb7", NULL,                /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *iso0a7[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xa1", "\xa1",              /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, "\xde", "\xfe", "\xde", "\xfe",          /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *iso0ff[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, "\xb7", NULL, NULL,                /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  "\xa2", "\xa3", "\xac", "\xaf", "\xa6", "\xa5", NULL, NULL,      /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *iso1d6[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, "\xdf", NULL, NULL, NULL, NULL,                /* c0 */
  NULL, NULL, NULL, NULL, NULL, "\xb5", NULL, NULL,                /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, "\xdf", NULL, NULL                 /* f8 */
};

static const char *iso1d7[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, "\xb5",                /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, "\xdf",                /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, "\xb5", NULL, NULL, NULL, NULL, NULL, NULL,                /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, "\xdf", NULL, NULL, NULL, NULL, NULL, NULL,                /* 70 */
  NULL, NULL, NULL, "\xb5", NULL, NULL, NULL, NULL,                /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, "\xdf", NULL, NULL, NULL, NULL,                /* a8 */
  NULL, NULL, NULL, NULL, NULL, "\xb5", NULL, NULL,                /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};


/* +++ ASCII-Sort-Tabellen +++ */

static const char *asort000[] = {
  "\00", " ", " ", " ", " ", " ", " ", " ",                        /* 00 */
  " ", " ", " ", " ", " ", " ", " ", " ",                          /* 08 */
  " ", " ", " ", " ", " ", " ", " ", " ",                          /* 10 */
  " ", " ", " ", " ", " ", " ", " ", " ",                          /* 18 */
  "\x20", "\x21", "\x22", "\x23", "\x24", "\x25", "\x26", "\x27",  /* 20 */
  "\x28", "\x29", "\x2a", "\x2b", "\x2c", "\x2d", "\x2e", "\x2f",  /* 28 */
  "\x30", "\x31", "\x32", "\x33", "\x34", "\x35", "\x36", "\x37",  /* 30 */
  "\x38", "\x39", "\x3a", "\x3b", "\x3c", "\x3d", "\x3e", "\x3f",  /* 38 */
  "\x40", "\x41", "\x42", "\x43", "\x44", "\x45", "\x46", "\x47",  /* 40 */
  "\x48", "\x49", "\x4a", "\x4b", "\x4c", "\x4d", "\x4e", "\x4f",  /* 48 */
  "\x50", "\x51", "\x52", "\x53", "\x54", "\x55", "\x56", "\x57",  /* 50 */
  "\x58", "\x59", "\x5a", "\x5b", "\x5c", "\x5d", "\x5e", "\x5f",  /* 58 */
  "\x60", "\x61", "\x62", "\x63", "\x64", "\x65", "\x66", "\x67",  /* 60 */
  "\x68", "\x69", "\x6a", "\x6b", "\x6c", "\x6d", "\x6e", "\x6f",  /* 68 */
  "\x70", "\x71", "\x72", "\x73", "\x74", "\x75", "\x76", "\x77",  /* 70 */
  "\x78", "\x79", "\x7a", "\x7b", "\x7c", "\x7d", "\x7e", "\x7f",  /* 78 */
  "\x80", NULL, "'", NULL, "\"", NULL, NULL, NULL,                 /* 80 - enh. */
  NULL, NULL, NULL, "<", "OE", NULL, NULL, NULL,                   /* 88 - enh. */
  NULL, "'", "'", "\"", "\"", NULL, NULL, NULL,                    /* 90 - enh. */
  "~", "TM", NULL, ">", "oe", NULL, NULL, NULL,                    /* 98 - enh. */
  " ", NULL, "\xa2", "\xa3", NULL, "\xa5", NULL, NULL,             /* a0 */
  NULL, NULL, NULL, "\"", NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, "+-", NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, "\"", NULL, NULL, NULL, NULL,                  /* b8 */
  "A", "A", "A", "A", "A?", "A", "AE", "C",                        /* c0 */
  "E", "E", "E", "E", "I", "I", "I", "I",                          /* c8 */
  "D", "N", "O", "O", "O", "O", "O?", NULL,                        /* d0 */
  "O", "U", "U", "U", "U?", "Y", "P", "ss",                        /* d8 */
  "a", "a", "a", "a", "a?", "a", "ae", "c",                        /* e0 */
  "e", "e", "e", "e", "i", "i", "i", "i",                          /* e8 */
  "d", "n", "o", "o", "o", "o", "o?", NULL,                        /* f0 */
  "o", "u", "u", "u", "u?", "y", "p", "y"                          /* f8 */
};

static const char *asort002[] = {
  "A", "a", "A", "a", "E", "e", "E", "e",                          /* 00 */
  "I", "i", "I", "i", "O", "o", "O", "o",                          /* 08 */
  "R", "r", "R", "r", "U", "u", "U", "u",                          /* 10 */
  "S", "s", "T", "t", "Z", "z", "H", "h",                          /* 18 */
  "N", "d", "OU", "ou", "Z", "z", "A", "a",                        /* 20 */
  "E", "e", "O?", "o?", "O", "o", "O", "o",                        /* 28 */
  "O", "o", "Y", "y", "l", "n", "t", "j",                          /* 30 */
  "db", "qp", "A", "C", "c", "L", "T", "s",                        /* 38 */
  "z", NULL, NULL, "B", "U", "V", "E", "e",                        /* 40 */
  "J", "j", "Q", "q", "R", "r", "Y", "y",                          /* 48 */
  "a", "a", "a", "b", NULL, "c", "d", "d",                         /* 50 */
  "e", "e", "e", "e", "e", "e", "e", "j",                          /* 58 */
  "g", "g", "G", "g", NULL, "h", "h", "h",                         /* 60 */
  "i", "i", "I", "l", "l", "l", "lz", "m",                         /* 68 */
  "m", "m", "n", "n", "N", "o", "OE", "o",                         /* 70 */
  NULL, "r", "r", "r", "r", "r", "r", "r",                         /* 78 */
  "R", "R", "s", "s", "j", "s", "s", "t",                          /* 80 */
  "t", "u", "y", "v", "v", "w", "y", "Y",                          /* 88 */
  "z", "z", "z", "z", NULL, NULL, NULL, "C",                       /* 90 */
  NULL, "B", "e", "G", "H", "j", "k", "L",                         /* 98 */
  "q", NULL, NULL, "dz", "dz", "dz", "ts", "ts",                   /* a0 */
  "tc", "fng", "ls", "lz", NULL, NULL, "h", "h",                   /* a8 */
  "h", "h", "j", "r", "r", "r", "R", "w",                          /* b0 */
  "y", NULL, NULL, "'", "'", NULL, NULL, NULL,                     /* b8 */
  NULL, NULL, "<", ">", NULL, NULL, "^", NULL,                     /* c0 */
  NULL, NULL, "'", "'", NULL, NULL, NULL, NULL,                    /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, "+", "-",                    /* d0 */
  NULL, NULL, NULL, NULL, "~", "\"", NULL, NULL,                   /* d8 */
  NULL, "l", "s", "x", NULL, NULL, NULL, NULL,                     /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\"", NULL,                  /* e8 */
  NULL, "<", ">", NULL, "'", "\"", "\"", "~",                      /* f0 */
  ":", NULL, NULL, NULL, NULL, NULL, NULL, NULL                    /* f8 */
};

static const char *asort020[] = {
  " ", " ", " ", " ", " ", " ", " ", " ",                          /* 00 */
  " ", " ", " ", " ", " ", " ", " ", " ",                          /* 08 */
  "-", "-", "-", "-", "-", "-", "||", "_",                         /* 10 */
  "'", "'", "'", "'", "\"", "\"", "\"", "\"",                      /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, " ",                   /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, "'", "'", NULL, "!!", NULL, NULL, NULL,                    /* 38 */
  NULL, "^", "*", "-", "/", "[", "]", "??",                        /* 40 */
  "?!", "!?", NULL, NULL, NULL, NULL, NULL, ";",                   /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, " ",                   /* 58 */
  NULL, NULL, NULL, " ", "+", NULL, NULL, NULL,                    /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  "0", "i", NULL, NULL, "4", "5", "6", "7",                        /* 70 */
  "8", "9", "+", "-", "=", "(", ")", "n",                          /* 78 */
  "0", "1", "2", "3", "4", "5", "6", "7",                          /* 80 */
  "8", "9", "+", "-", "=", "(", ")", NULL,                         /* 88 */
  "a", "e", "o", "x", "e", "h", "k", "l",                          /* 90 */
  "m", "n", "p", "s", "t", NULL, NULL, NULL,                       /* 98 */
  "\x80", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                /* a0 */
  NULL, NULL, NULL, NULL, "\x80", NULL, NULL, NULL,                /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, "\\", "||", NULL,                  /* e0 */
  NULL, NULL, NULL, "//", NULL, NULL, NULL, NULL,                  /* e8 */
  "*", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                   /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asort021[] = {
  NULL, NULL, "C", NULL, NULL, NULL, NULL, NULL,                   /* 00 */
  NULL, NULL, "g", "H", "H", "H", "h", "h",                        /* 08 */
  "I", "I", "L", "l", "lb", "N", "No", "(P)",                      /* 10 */
  "P", "P", "Q", "R", "R", "R", NULL, NULL,                        /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  "Z", "i", "K", "A", "B", "C", "e", "e",                          /* 28 */
  "E", "F", "F", "M", "o", NULL, NULL, NULL,                       /* 30 */
  NULL, NULL, "Q", NULL, NULL, NULL, NULL, NULL,                   /* 38 */
  NULL, "G", "L", "L", "Y", "D", "d", "e",                         /* 40 */
  "i", "j", NULL, "&", NULL, NULL, "F", NULL,                      /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  "I", "II", "III", "IV", "V", "VI", "VII", "VIII",                /* 60 */
  "IX", "X", "XI", "XII", "L", "C", "D", "M",                      /* 68 */
  "i", "ii", "iii", "iv", "v", "vi", "vii", "viii",                /* 70 */
  "ix", "x", "xi", "xii", "l", "c", "d", "m",                      /* 78 */
  "CD", NULL, NULL, "C", "c", NULL, NULL, NULL,                    /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asort023[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, "#",                   /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, "<", ">", NULL, NULL, NULL, NULL, NULL,                    /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  "~", ">", ",", NULL, NULL, NULL, ";", NULL,                      /* 68 */
  NULL, NULL, NULL, "i", "p", NULL, "a", "e",                      /* 70 */
  "i", NULL, "a", NULL, NULL, NULL, NULL, NULL,                    /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, "(", "(", "(", ")", ")",                       /* 98 */
  ")", "[", "[", "[", "]", "]", "]", "{",                          /* a0 */
  "{", "{", "|", "}", "}", "}", "|", NULL,                         /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asort02c[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  "L", "l", "L", "P", "R", "a", "t", "H",                          /* 60 */
  "h", "K", "k", "Z", "z", "A", "M", "A",                          /* 68 */
  "A", "v", "W", "w", "v", "H", "h", NULL,                         /* 70 */
  "e", "r", "o", "E", "j", "V", "S", "Z",                          /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asort0ff[] = {
  " ", "!", "\"", "#", "$", "%", "&", "'",                         /* 00 */
  "(", ")", "*", "+", ",", "-", ".", "/",                          /* 08 */
  "0", "1", "2", "3", "4", "5", "6", "7",                          /* 10 */
  "8", "9", ":", ";", "<", "=", ">", "?",                          /* 18 */
  "@", "A", "B", "C", "D", "E", "F", "G",                          /* 20 */
  "H", "I", "J", "K", "L", "M", "N", "O",                          /* 28 */
  "P", "Q", "R", "S", "T", "U", "V", "W",                          /* 30 */
  "X", "Y", "Z", "[", "\\", "]", "^", "_",                         /* 38 */
  "`", "a", "b", "c", "d", "e", "f", "g",                          /* 40 */
  "h", "i", "j", "k", "l", "m", "n", "o",                          /* 48 */
  "p", "q", "r", "s", "t", "u", "v", "w",                          /* 50 */
  "x", "y", "z", "{", "|", "}", "~", "(",                          /* 58 */
  ")", ".", NULL, NULL, ",", NULL, NULL, NULL,                     /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  "\xa2", "\xa3", NULL, NULL, NULL, "\xa5", NULL, NULL,            /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asort1d5[] = {
  "w", "x", "y", "z", "A", "B", "C", "D",                          /* 00 */
  "E", "F", "G", "H", "I", "J", "K", "L",                          /* 08 */
  "M", "N", "O", "P", "Q", "R", "S", "T",                          /* 10 */
  "U", "V", "W", "X", "Y", "Z", "a", "b",                          /* 18 */
  "c", "d", "e", "f", "g", "h", "i", "j",                          /* 20 */
  "k", "l", "m", "n", "o", "p", "q", "r",                          /* 28 */
  "s", "t", "u", "v", "w", "x", "y", "z",                          /* 30 */
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* 38 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* 40 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* 48 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* 50 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* 58 */
  "o", "p", "q", "r", "s", "t", "u", "v",                          /* 60 */
  "w", "x", "y", "z", "A", "B", "C", "D",                          /* 68 */
  "E", "F", "G", "H", "I", "J", "K", "L",                          /* 70 */
  "M", "N", "O", "P", "Q", "R", "S", "T",                          /* 78 */
  "U", "V", "W", "X", "Y", "Z", "a", "b",                          /* 80 */
  "c", "d", "e", "f", "g", "h", "i", "j",                          /* 88 */
  "k", "l", "m", "n", "o", "p", "q", "r",                          /* 90 */
  "s", "t", "u", "v", "w", "x", "y", "z",                          /* 98 */
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* a0 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* a8 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* b0 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* b8 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* c0 */
  "o", "p", "q", "r", "s", "t", "u", "v",                          /* c8 */
  "w", "x", "y", "z", "A", "B", "C", "D",                          /* d0 */
  "E", "F", "G", "H", "I", "J", "K", "L",                          /* d8 */
  "M", "N", "O", "P", "Q", "R", "S", "T",                          /* e0 */
  "U", "V", "W", "X", "Y", "Z", "a", "b",                          /* e8 */
  "c", "d", "e", "f", "g", "h", "i", "j",                          /* f0 */
  "k", "l", "m", "n", "o", "p", "q", "r"                           /* f8 */
};

static const char *asort1d6[] = {
  "s", "t", "u", "v", "w", "x", "y", "z",                          /* 00 */
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* 08 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* 10 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* 18 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* 20 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* 28 */
  "o", "p", "q", "r", "s", "t", "u", "v",                          /* 30 */
  "w", "x", "y", "z", "A", "B", "C", "D",                          /* 38 */
  "E", "F", "G", "H", "I", "J", "K", "L",                          /* 40 */
  "M", "N", "O", "P", "Q", "R", "S", "T",                          /* 48 */
  "U", "V", "W", "X", "Y", "Z", "a", "b",                          /* 50 */
  "c", "d", "e", "f", "g", "h", "i", "j",                          /* 58 */
  "k", "l", "m", "n", "o", "p", "q", "r",                          /* 60 */
  "s", "t", "u", "v", "w", "x", "y", "z",                          /* 68 */
  "A", "B", "C", "D", "E", "F", "G", "H",                          /* 70 */
  "I", "J", "K", "L", "M", "N", "O", "P",                          /* 78 */
  "Q", "R", "S", "T", "U", "V", "W", "X",                          /* 80 */
  "Y", "Z", "a", "b", "c", "d", "e", "f",                          /* 88 */
  "g", "h", "i", "j", "k", "l", "m", "n",                          /* 90 */
  "o", "p", "q", "r", "s", "t", "u", "v",                          /* 98 */
  "w", "x", "y", "z", "i", "j", NULL, NULL,                        /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *asort1d7[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, "0", "1",                    /* c8 */
  "2", "3", "4", "5", "6", "7", "8", "9",                          /* d0 */
  "0", "1", "2", "3", "4", "5", "6", "7",                          /* d8 */
  "8", "9", "0", "1", "2", "3", "4", "5",                          /* e0 */
  "6", "7", "8", "9", "0", "1", "2", "3",                          /* e8 */
  "4", "5", "6", "7", "8", "9", "0", "1",                          /* f0 */
  "2", "3", "4", "5", "6", "7", "8", "9"                           /* f8 */
};


/* +++ Uppercase-Tabellen +++ */

static const char *upcase000[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, "A", "B", "C", "D", "E", "F", "G",                         /* 60 */
  "H", "I", "J", "K", "L", "M", "N", "O",                          /* 68 */
  "P", "Q", "R", "S", "T", "U", "V", "W",                          /* 70 */
  "X", "Y", "Z", NULL, NULL, NULL, NULL, NULL,                     /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, "\xe1\xba\x9e",        /* d8 */
  "\xc3\x80", "\xc3\x81", "\xc3\x82", "\xc3\x83", "\xc3\x84", "\xc3\x85", "\xc3\x86", "\xc3\x87",  /* e0 */
  "\xc3\x88", "\xc3\x89", "\xc3\x8a", "\xc3\x8b", "\xc3\x8c", "\xc3\x8d", "\xc3\x8e", "\xc3\x8f",  /* e8 */
  "\xc3\x90", "\xc3\x91", "\xc3\x92", "\xc3\x93", "\xc3\x94", "\xc3\x95", "\xc3\x96", NULL,        /* f0 */
  "\xc3\x98", "\xc3\x99", "\xc3\x9a", "\xc3\x9b", "\xc3\x9c", "\xc3\x9d", "\xc3\x9e", "\xc5\xb8"   /* f8 */
};

static const char *upcase000_iso[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, "A", "B", "C", "D", "E", "F", "G",                         /* 60 */
  "H", "I", "J", "K", "L", "M", "N", "O",                          /* 68 */
  "P", "Q", "R", "S", "T", "U", "V", "W",                          /* 70 */
  "X", "Y", "Z", NULL, NULL, NULL, NULL, NULL,                     /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  "\xc0", "\xc1", "\xc2", "\xc3", "\xc4", "\xc5", "\xc6", "\xc7",  /* e0 */
  "\xc8", "\xc9", "\xca", "\xcb", "\xcc", "\xcd", "\xce", "\xcf",  /* e8 */
  "\xd0", "\xd1", "\xd2", "\xd3", "\xd4", "\xd5", "\xd6", NULL,    /* f0 */
  "\xd8", "\xd9", "\xda", "\xdb", "\xdc", "\xdd", "\xde", NULL     /* f8 */
};

static const char *upcase001[] = {
  NULL, "\xc4\x80", NULL, "\xc4\x82", NULL, "\xc4\x84", NULL, "\xc4\x86",      /* 00 */
  NULL, "\xc4\x88", NULL, "\xc4\x8a", NULL, "\xc4\x8c", NULL, "\xc4\x8e",      /* 08 */
  NULL, "\xc4\x90", NULL, "\xc4\x92", NULL, "\xc4\x94", NULL, "\xc4\x96",      /* 10 */
  NULL, "\xc4\x98", NULL, "\xc4\x9a", NULL, "\xc4\x9c", NULL, "\xc4\x9e",      /* 18 */
  NULL, "\xc4\xa0", NULL, "\xc4\xa2", NULL, "\xc4\xa4", NULL, "\xc4\xa6",      /* 20 */
  NULL, "\xc4\xa8", NULL, "\xc4\xaa", NULL, "\xc4\xac", NULL, "\xc4\xae",      /* 28 */
  NULL, NULL, NULL, "\xc4\xb2", NULL, "\xc4\xb4", NULL, "\xc4\xb6",            /* 30 */
  NULL, NULL, "\xc4\xb9", NULL, "\xc4\xbb", NULL, "\xc4\xbd", NULL,            /* 38 */
  "\xc4\xbf", NULL, "\xc5\x81", NULL, "\xc5\x83", NULL, "\xc5\x85", NULL,      /* 40 */
  "\xc5\x87", NULL, NULL, "\xc5\x8a", NULL, "\xc5\x8c", NULL, "\xc5\x8e",      /* 48 */
  NULL, "\xc5\x90", NULL, "\xc5\x92", NULL, "\xc5\x94", NULL, "\xc5\x96",      /* 50 */
  NULL, "\xc5\x98", NULL, "\xc5\x9a", NULL, "\xc5\x9c", NULL, "\xc5\x9e",      /* 58 */
  NULL, "\xc5\xa0", NULL, "\xc5\xa2", NULL, "\xc5\xa4", NULL, "\xc5\xa6",      /* 60 */
  NULL, "\xc5\xa8", NULL, "\xc5\xaa", NULL, "\xc5\xac", NULL, "\xc5\xae",      /* 68 */
  NULL, "\xc5\xb0", NULL, "\xc5\xb2", NULL, "\xc5\xb4", NULL, "\xc5\xb6",      /* 70 */
  NULL, NULL, "\xc5\xb9", NULL, "\xc5\xbb", NULL, "\xc5\xbd", NULL,            /* 78 */
  NULL, NULL, NULL, "\xc6\x82", NULL, "\xc6\x84", NULL, NULL,                  /* 80 */
  "\xc6\x87", NULL, NULL, NULL, "\xc6\x8b", NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, "\xc6\x91", NULL, NULL, NULL, NULL, NULL,                        /* 90 */                                                                           
  NULL, "\xc6\x98", NULL, NULL, NULL, NULL, NULL, NULL,                        /* 98 */                                                                           
  NULL, "\xc6\xa0", NULL, "\xc6\xa2", NULL, "\xc6\xa4", NULL, NULL,            /* a0 */                                                                           
  "\xc6\xa7", NULL, NULL, NULL, NULL, "\xc6\xac", NULL, NULL,                  /* a8 */                                                                           
  "\xc6\xaf", NULL, NULL, NULL, "\xc6\xb3", NULL, "\xc6\xb5", NULL,            /* b0 */                                                                           
  NULL, "\xc6\xb8", NULL, NULL, NULL, "\xc6\xbc", NULL, NULL,                  /* b8 */                                                                           
  NULL, NULL, NULL, NULL, NULL, NULL, "\xc7\x84", NULL,                        /* c0 */                                                                           
  NULL, "\xc7\x87", NULL, NULL, "\xc7\x8a", NULL, "\xc7\x8d", NULL,            /* c8 */                                                                           
  "\xc7\x8f", NULL, "\xc7\x91", NULL, "\xc7\x93", NULL, "\xc7\x95", NULL,      /* d0 */                                                                           
  "\xc7\x97", NULL, "\xc7\x99", NULL, "\xc7\x9b", NULL, NULL, "\xc7\x9e",      /* d8 */                                                                           
  NULL, "\xc7\xa0", NULL, "\xc7\xa2", NULL, "\xc7\xa4", NULL, "\xc7\xa6",      /* e0 */                                                                           
  NULL, "\xc7\xa8", NULL, "\xc7\xaa", NULL, "\xc7\xac", NULL, "\xc7\xae",      /* e8 */
  NULL, NULL, NULL, "\xc7\xb1", NULL, "\xc7\xb4", NULL, NULL,                  /* f0 */
  NULL, "\xc7\xb8", NULL, "\xc7\xba", NULL, "\xc7\xbc", NULL, "\xc7\xbe"       /* f8 */
};

static const char *upcase002[] = {
  NULL, "\xc8\x80", NULL, "\xc8\x82", NULL, "\xc8\x84", NULL, "\xc8\x86",      /* 00 */
  NULL, "\xc8\x88", NULL, "\xc8\x8a", NULL, "\xc8\x8c", NULL, "\xc8\x8e",      /* 08 */
  NULL, "\xc8\x90", NULL, "\xc8\x92", NULL, "\xc8\x94", NULL, "\xc8\x96",      /* 10 */
  NULL, "\xc8\x98", NULL, "\xc8\x9a", NULL, "\xc8\x9c", NULL, "\xc8\x9e",      /* 18 */
  NULL, NULL, NULL, "\xc8\xa2", NULL, "\xc8\xa4", NULL, "\xc8\xa6",            /* 20 */
  NULL, "\xc8\xa8", NULL, "\xc8\xaa", NULL, "\xc8\xac", NULL, "\xc8\xae",      /* 28 */
  NULL, "\xc8\xb0", NULL, "\xc8\xb2", NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, "\xc8\xbb", NULL, NULL, NULL,                        /* 38 */
  NULL, NULL, "\xc9\x81", NULL, NULL, NULL, NULL, "\xc9\x86",                  /* 40 */
  NULL, "\xc9\x88", NULL, "\xc9\x8a", NULL, "\xc9\x8c", NULL, "\xc9\x8e",      /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                               /* f8 */
};

static const char *upcase01e[] = {
  NULL, "\xe1\xb8\x80", NULL, "\xe1\xb8\x82", NULL, "\xe1\xb8\x84", NULL, "\xe1\xb8\x86",      /* 00 */
  NULL, "\xe1\xb8\x88", NULL, "\xe1\xb8\x8a", NULL, "\xe1\xb8\x8c", NULL, "\xe1\xb8\x8e",      /* 08 */
  NULL, "\xe1\xb8\x90", NULL, "\xe1\xb8\x92", NULL, "\xe1\xb8\x94", NULL, "\xe1\xb8\x96",      /* 10 */
  NULL, "\xe1\xb8\x98", NULL, "\xe1\xb8\x9a", NULL, "\xe1\xb8\x9c", NULL, "\xe1\xb8\x9e",      /* 18 */
  NULL, "\xe1\xb8\xa0", NULL, "\xe1\xb8\xa2", NULL, "\xe1\xb8\xa4", NULL, "\xe1\xb8\xa6",      /* 20 */
  NULL, "\xe1\xb8\xa8", NULL, "\xe1\xb8\xaa", NULL, "\xe1\xb8\xac", NULL, "\xe1\xb8\xae",      /* 28 */
  NULL, "\xe1\xb8\xb0", NULL, "\xe1\xb8\xb2", NULL, "\xe1\xb8\xb4", NULL, "\xe1\xb8\xb6",      /* 30 */
  NULL, "\xe1\xb8\xb8", NULL, "\xe1\xb8\xba", NULL, "\xe1\xb8\xbc", NULL, "\xe1\xb8\xbe",      /* 38 */
  NULL, "\xe1\xb9\x80", NULL, "\xe1\xb9\x82", NULL, "\xe1\xb9\x84", NULL, "\xe1\xb9\x86",      /* 40 */
  NULL, "\xe1\xb9\x88", NULL, "\xe1\xb9\x8a", NULL, "\xe1\xb9\x8c", NULL, "\xe1\xb8\x9e",      /* 48 */
  NULL, "\xe1\xb9\x90", NULL, "\xe1\xb9\x92", NULL, "\xe1\xb9\x94", NULL, "\xe1\xb9\x96",      /* 50 */
  NULL, "\xe1\xb9\x98", NULL, "\xe1\xb9\x9a", NULL, "\xe1\xb9\x9c", NULL, "\xe1\xb8\x9e",      /* 58 */
  NULL, "\xe1\xb9\xa0", NULL, "\xe1\xb9\xa2", NULL, "\xe1\xb9\xa4", NULL, "\xe1\xb9\xa6",      /* 60 */
  NULL, "\xe1\xb9\xa8", NULL, "\xe1\xb9\xaa", NULL, "\xe1\xb9\xac", NULL, "\xe1\xb8\xae",      /* 68 */
  NULL, "\xe1\xb9\xb0", NULL, "\xe1\xb9\xb2", NULL, "\xe1\xb9\xb4", NULL, "\xe1\xb9\xb6",      /* 70 */
  NULL, "\xe1\xb9\xb8", NULL, "\xe1\xb9\xba", NULL, "\xe1\xb9\xbc", NULL, "\xe1\xb8\xbe",      /* 78 */
  NULL, "\xe1\xba\x80", NULL, "\xe1\xba\x82", NULL, "\xe1\xba\x84", NULL, "\xe1\xba\x86",      /* 80 */
  NULL, "\xe1\xba\x88", NULL, "\xe1\xba\x8a", NULL, "\xe1\xba\x8c", NULL, "\xe1\xba\x9e",      /* 88 */
  NULL, "\xe1\xba\x90", NULL, "\xe1\xba\x92", NULL, "\xe1\xba\x94", NULL, NULL,                /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                              /* 98 */
  NULL, "\xe1\xba\xa0", NULL, "\xe1\xba\xa2", NULL, "\xe1\xba\xa4", NULL, "\xe1\xba\xa6",      /* a0 */
  NULL, "\xe1\xba\xa8", NULL, "\xe1\xba\xaa", NULL, "\xe1\xba\xac", NULL, "\xe1\xba\xae",      /* a8 */
  NULL, "\xe1\xba\xb0", NULL, "\xe1\xba\xb2", NULL, "\xe1\xba\xb4", NULL, "\xe1\xba\xb6",      /* b0 */
  NULL, "\xe1\xba\xb8", NULL, "\xe1\xba\xba", NULL, "\xe1\xba\xbc", NULL, "\xe1\xba\xbe",      /* b8 */
  NULL, "\xe1\xbb\x80", NULL, "\xe1\xbb\x82", NULL, "\xe1\xbb\x84", NULL, "\xe1\xbb\x86",      /* c0 */
  NULL, "\xe1\xbb\x88", NULL, "\xe1\xbb\x8a", NULL, "\xe1\xbb\x8c", NULL, "\xe1\xbb\x9e",      /* c8 */
  NULL, "\xe1\xbb\x90", NULL, "\xe1\xbb\x92", NULL, "\xe1\xbb\x94", NULL, "\xe1\xbb\x96",      /* d0 */
  NULL, "\xe1\xbb\x98", NULL, "\xe1\xbb\x9a", NULL, "\xe1\xbb\x9c", NULL, "\xe1\xbb\x9e",      /* d8 */
  NULL, "\xe1\xbb\xa0", NULL, "\xe1\xbb\xa2", NULL, "\xe1\xbb\xa4", NULL, "\xe1\xbb\xa6",      /* e0 */
  NULL, "\xe1\xbb\xa8", NULL, "\xe1\xbb\xaa", NULL, "\xe1\xbb\xac", NULL, "\xe1\xbb\xae",      /* e8 */
  NULL, "\xe1\xbb\xb0", NULL, "\xe1\xbb\xb2", NULL, "\xe1\xbb\xb4", NULL, "\xe1\xbb\xb6",      /* f0 */
  NULL, "\xe1\xbb\xb8", NULL, "\xe1\xbb\xba", NULL, "\xe1\xbb\xbc", NULL, "\xe1\xbb\xbe"       /* f8 */
};

static const char *upcase02c[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 58 */
  NULL, "\xe2\xb1\xa0", NULL, NULL, NULL, NULL, NULL, NULL,                      /* 60 */
  "\xe2\xb1\xa7", NULL, "\xe2\xb1\xa9", NULL, "\xe2\xb1\xab", NULL, NULL, NULL,  /* 68 */
  NULL, NULL, NULL, "\xe2\xb1\xb2", NULL, NULL, NULL, NULL,                      /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 88 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 90 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 98 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* a0 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* a8 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* b0 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* b8 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* c0 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* c8 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* d0 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* d8 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* e0 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* e8 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* f0 */      
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                                 /* f8 */      
};

static const char *upcase0a7[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 18 */
  NULL, NULL, NULL, "\xea\x9c\xa2", NULL, "\xea\x9c\xa4", NULL, "\xea\x9c\xa6",            /* 20 */
  NULL, "\xea\x9c\xa8", NULL, "\xea\x9c\xaa", NULL, "\xea\x9c\xac", NULL, "\xea\x9c\xae",  /* 28 */
  NULL, NULL, NULL, "\xea\x9c\xb2", NULL, "\xea\x9c\xb4", NULL, "\xea\x9c\xb6",            /* 30 */
  NULL, "\xea\x9c\xb8", NULL, "\xea\x9c\xba", NULL, "\xea\x9c\xbc", NULL, "\xea\x9c\xbe",  /* 38 */
  NULL, "\xea\x9d\x80", NULL, "\xea\x9d\x82", NULL, "\xea\x9d\x84", NULL, "\xea\x9d\x86",  /* 40 */
  NULL, "\xea\x9d\x88", NULL, "\xea\x9d\x8a", NULL, "\xea\x9d\x8c", NULL, "\xea\x9d\x8e",  /* 48 */
  NULL, "\xea\x9d\x90", NULL, "\xea\x9d\x92", NULL, "\xea\x9d\x94", NULL, "\xea\x9d\x96",  /* 50 */
  NULL, "\xea\x9d\x98", NULL, "\xea\x9d\x9a", NULL, "\xea\x9d\x9c", NULL, "\xea\x9d\x9e",  /* 58 */
  NULL, "\xea\x9d\xa0", NULL, "\xea\x9d\xa2", NULL, "\xea\x9d\xa4", NULL, "\xea\x9d\xa6",  /* 60 */
  NULL, "\xea\x9d\xa8", NULL, "\xea\x9d\xaa", NULL, "\xea\x9d\xac", NULL, "\xea\x9d\xae",  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 70 */
  NULL, NULL, "\xea\x9d\xb9", NULL, "\xea\x9d\xbb", NULL, NULL, "\xea\x9d\xbe",            /* 78 */
  NULL, "\xea\x9e\x80", NULL, "\xea\x9e\x82", NULL, "\xea\x9e\x84", NULL, "\xea\x9e\x86",  /* 80 */
  NULL, NULL, NULL, NULL, "\xea\x9e\x8b", NULL, NULL, NULL,                                /* 88 */
  NULL, "\xea\x9e\x90", NULL, NULL, NULL, NULL, NULL, NULL,                                /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 98 */
  NULL, "\xea\x9e\xa0", NULL, "\xea\x9e\xa2", NULL, "\xea\x9e\xa4", NULL, "\xea\x9e\xa6",  /* a0 */
  NULL, "\xea\x9e\xa8", NULL, NULL, NULL, NULL, NULL, NULL,                                /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                                           /* f8 */
};

static const char *upcase0ff[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, "\xef\xbc\xa1", "\xef\xbc\xa2", "\xef\xbc\xa3", "\xef\xbc\xa4", "\xef\xbc\xa5", "\xef\xbc\xa6", "\xef\xbc\xa7",            /* 40 */
  "\xef\xbc\xa8", "\xef\xbc\xa9", "\xef\xbc\xaa", "\xef\xbc\xab", "\xef\xbc\xac", "\xef\xbc\xad", "\xef\xbc\xae", "\xef\xbc\xaf",  /* 48 */
  "\xef\xbc\xb0", "\xef\xbc\xb1", "\xef\xbc\xb2", "\xef\xbc\xb3", "\xef\xbc\xb4", "\xef\xbc\xb5", "\xef\xbc\xb6", "\xef\xbc\xb7",  /* 50 */
  "\xef\xbc\xb8", "\xef\xbc\xb9", "\xef\xbc\xba", NULL, NULL, NULL, NULL, NULL,                                                    /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */                                                        
};

static const char *upcase1d4[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 10 */
  NULL, NULL, "\xf0\x9d\x90\x80", "\xf0\x9d\x90\x81", "\xf0\x9d\x90\x82", "\xf0\x9d\x90\x83", "\xf0\x9d\x90\x84", "\xf0\x9d\x90\x85",                             /* 18 */
  "\xf0\x9d\x90\x86", "\xf0\x9d\x90\x87", "\xf0\x9d\x90\x88", "\xf0\x9d\x90\x89", "\xf0\x9d\x90\x8a", "\xf0\x9d\x90\x8b", "\xf0\x9d\x90\x8c", "\xf0\x9d\x90\x8d", /* 20 */
  "\xf0\x9d\x90\x8e", "\xf0\x9d\x90\x8f", "\xf0\x9d\x90\x90", "\xf0\x9d\x90\x91", "\xf0\x9d\x90\x92", "\xf0\x9d\x90\x93", "\xf0\x9d\x90\x94", "\xf0\x9d\x90\x95", /* 28 */
  "\xf0\x9d\x90\x96", "\xf0\x9d\x90\x97", "\xf0\x9d\x90\x98", "\xf0\x9d\x90\x99", NULL, NULL, NULL, NULL,                                                         /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xf0\x9d\x90\xb4", "\xf0\x9d\x90\xb5",                                                                                     /* 48 */
  "\xf0\x9d\x90\xb6", "\xf0\x9d\x90\xb7", "\xf0\x9d\x90\xb8", "\xf0\x9d\x90\xb9", "\xf0\x9d\x90\xba", "\xf0\x9d\x90\xbb", "\xf0\x9d\x90\xbc", "\xf0\x9d\x90\xbd", /* 50 */
  "\xf0\x9d\x90\xbe", "\xf0\x9d\x90\xbf", "\xf0\x9d\x91\x80", "\xf0\x9d\x91\x81", "\xf0\x9d\x91\x82", "\xf0\x9d\x91\x83", "\xf0\x9d\x91\x84", "\xf0\x9d\x91\x85", /* 58 */
  "\xf0\x9d\x91\x86", "\xf0\x9d\x91\x87", "\xf0\x9d\x91\x88", "\xf0\x9d\x91\x89", "\xf0\x9d\x91\x8a", "\xf0\x9d\x91\x8b", "\xf0\x9d\x91\x8c", "\xf0\x9d\x91\x8d", /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 78 */
  NULL, NULL, "\xf0\x9d\x91\xa8", "\xf0\x9d\x91\xa9", "\xf0\x9d\x91\xaa", "\xf0\x9d\x91\xab", "\xf0\x9d\x91\xac", "\xf0\x9d\x91\xad",                             /* 80 */
  "\xf0\x9d\x91\xae", "\xf0\x9d\x91\xaf", "\xf0\x9d\x91\xb0", "\xf0\x9d\x91\xb1", "\xf0\x9d\x91\xb2", "\xf0\x9d\x91\xb3", "\xf0\x9d\x91\xb4", "\xf0\x9d\x91\xb5", /* 88 */
  "\xf0\x9d\x91\xb6", "\xf0\x9d\x91\xb7", "\xf0\x9d\x91\xb8", "\xf0\x9d\x91\xb9", "\xf0\x9d\x91\xba", "\xf0\x9d\x91\xbb", "\xf0\x9d\x91\xbc", "\xf0\x9d\x91\xbd", /* 90 */
  "\xf0\x9d\x91\xbe", "\xf0\x9d\x91\xbf", "\xf0\x9d\x92\x80", "\xf0\x9d\x92\x81", NULL, NULL, NULL, NULL,                                                         /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xf0\x9d\x92\x9c", "\xf0\x9d\x92\x9d",                                                                                     /* b0 */
  "\xf0\x9d\x92\x9e", "\xf0\x9d\x92\x9f", "\xf0\x9d\x92\xa0", "\xf0\x9d\x92\xa1", "\xf0\x9d\x92\xa2", "\xf0\x9d\x92\xa3", "\xf0\x9d\x92\xa4", "\xf0\x9d\x92\xa5", /* b8 */
  "\xf0\x9d\x92\xa6", "\xf0\x9d\x92\xa7", "\xf0\x9d\x92\xa8", "\xf0\x9d\x92\xa9", "\xf0\x9d\x92\xaa", "\xf0\x9d\x92\xab", "\xf0\x9d\x92\xac", "\xf0\x9d\x92\xad", /* c0 */
  "\xf0\x9d\x92\xae", "\xf0\x9d\x92\xaf", "\xf0\x9d\x92\xb0", "\xf0\x9d\x92\xb1", "\xf0\x9d\x92\xb2", "\xf0\x9d\x92\xb3", "\xf0\x9d\x92\xb4", "\xf0\x9d\x92\xb5", /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* e0 */
  NULL, NULL, "\xf0\x9d\x93\x90", "\xf0\x9d\x93\x91", "\xf0\x9d\x93\x92", "\xf0\x9d\x93\x93", "\xf0\x9d\x93\x94", "\xf0\x9d\x93\x95",                             /* e8 */
  "\xf0\x9d\x93\x96", "\xf0\x9d\x93\x97", "\xf0\x9d\x93\x98", "\xf0\x9d\x93\x99", "\xf0\x9d\x93\x9a", "\xf0\x9d\x93\x9b", "\xf0\x9d\x93\x9c", "\xf0\x9d\x93\x9d", /* f0 */
  "\xf0\x9d\x93\x9e", "\xf0\x9d\x93\x9f", "\xf0\x9d\x93\xa0", "\xf0\x9d\x93\xa1", "\xf0\x9d\x93\xa2", "\xf0\x9d\x93\xa3", "\xf0\x9d\x93\xa4", "\xf0\x9d\x93\xa5"  /* f8 */
};

static const char *upcase1d5[] = {
  "\xf0\x9d\x93\xa6", "\xf0\x9d\x93\xa7", "\xf0\x9d\x93\xa8", "\xf0\x9d\x93\xa9", NULL, NULL, NULL, NULL,                                                         /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xf0\x9d\x94\x84", "\xf0\x9d\x94\x85",                                                                                     /* 18 */
  "\xf0\x9d\x94\x86", "\xf0\x9d\x94\x87", "\xf0\x9d\x94\x88", "\xf0\x9d\x94\x89", "\xf0\x9d\x94\x8a", "\xf0\x9d\x94\x8b", "\xf0\x9d\x94\x8c", "\xf0\x9d\x94\x8d", /* 20 */
  "\xf0\x9d\x94\x8e", "\xf0\x9d\x94\x8f", "\xf0\x9d\x94\x90", "\xf0\x9d\x94\x91", "\xf0\x9d\x94\x92", "\xf0\x9d\x94\x93", "\xf0\x9d\x94\x94", "\xf0\x9d\x94\x95", /* 28 */
  "\xf0\x9d\x94\x96", "\xf0\x9d\x94\x97", "\xf0\x9d\x94\x98", "\xf0\x9d\x94\x99", "\xf0\x9d\x94\x9a", "\xf0\x9d\x94\x9b", "\xf0\x9d\x94\x9c", "\xf0\x9d\x94\x9d", /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 48 */
  NULL, NULL, "\xf0\x9d\x94\xb8", "\xf0\x9d\x94\xb9", "\xf0\x9d\x94\xba", "\xf0\x9d\x94\xbb", "\xf0\x9d\x94\xbc", "\xf0\x9d\x94\xbd",                             /* 50 */
  "\xf0\x9d\x94\xbe", "\xf0\x9d\x94\xbf", "\xf0\x9d\x95\x80", "\xf0\x9d\x95\x81", "\xf0\x9d\x95\x82", "\xf0\x9d\x95\x83", "\xf0\x9d\x95\x84", "\xf0\x9d\x95\x85", /* 58 */
  "\xf0\x9d\x95\x86", "\xf0\x9d\x95\x87", "\xf0\x9d\x95\x88", "\xf0\x9d\x95\x89", "\xf0\x9d\x95\x8a", "\xf0\x9d\x95\x8b", "\xf0\x9d\x95\x8c", "\xf0\x9d\x95\x8d", /* 60 */
  "\xf0\x9d\x95\x8e", "\xf0\x9d\x95\x8f", "\xf0\x9d\x95\x90", "\xf0\x9d\x95\x91", NULL, NULL, NULL, NULL,                                                         /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xf0\x9d\x95\xac", "\xf0\x9d\x95\xad",                                                                                     /* 80 */
  "\xf0\x9d\x95\xae", "\xf0\x9d\x95\xaf", "\xf0\x9d\x95\xb0", "\xf0\x9d\x95\xb1", "\xf0\x9d\x95\xb2", "\xf0\x9d\x95\xb3", "\xf0\x9d\x95\xb4", "\xf0\x9d\x95\xb5", /* 88 */
  "\xf0\x9d\x95\xb6", "\xf0\x9d\x95\xb7", "\xf0\x9d\x95\xb8", "\xf0\x9d\x95\xb9", "\xf0\x9d\x95\xba", "\xf0\x9d\x95\xbb", "\xf0\x9d\x95\xbc", "\xf0\x9d\x95\xbd", /* 90 */
  "\xf0\x9d\x95\xbe", "\xf0\x9d\x95\xbf", "\xf0\x9d\x96\x80", "\xf0\x9d\x96\x81", "\xf0\x9d\x96\x82", "\xf0\x9d\x96\x83", "\xf0\x9d\x96\x84", "\xf0\x9d\x96\x85", /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* b0 */
  NULL, NULL, "\xf0\x9d\x96\xa0", "\xf0\x9d\x96\xa1", "\xf0\x9d\x96\xa2", "\xf0\x9d\x96\xa3", "\xf0\x9d\x96\xa4", "\xf0\x9d\x96\xa5",                             /* b8 */
  "\xf0\x9d\x96\xa6", "\xf0\x9d\x96\xa7", "\xf0\x9d\x96\xa8", "\xf0\x9d\x96\xa9", "\xf0\x9d\x96\xaa", "\xf0\x9d\x96\xab", "\xf0\x9d\x96\xac", "\xf0\x9d\x96\xad", /* c0 */
  "\xf0\x9d\x96\xae", "\xf0\x9d\x96\xaf", "\xf0\x9d\x96\xb0", "\xf0\x9d\x96\xb1", "\xf0\x9d\x96\xb2", "\xf0\x9d\x96\xb3", "\xf0\x9d\x96\xb4", "\xf0\x9d\x96\xb5", /* c8 */
  "\xf0\x9d\x96\xb6", "\xf0\x9d\x96\xb7", "\xf0\x9d\x96\xb8", "\xf0\x9d\x96\xb9", NULL, NULL, NULL, NULL,                                                         /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xf0\x9d\x97\x94", "\xf0\x9d\x97\x95",                                                                                     /* e8 */
  "\xf0\x9d\x97\x96", "\xf0\x9d\x97\x97", "\xf0\x9d\x97\x98", "\xf0\x9d\x97\x99", "\xf0\x9d\x97\x9a", "\xf0\x9d\x97\x9b", "\xf0\x9d\x97\x9c", "\xf0\x9d\x97\x9d", /* f0 */
  "\xf0\x9d\x97\x9e", "\xf0\x9d\x97\x9f", "\xf0\x9d\x97\xa0", "\xf0\x9d\x97\xa1", "\xf0\x9d\x97\xa2", "\xf0\x9d\x97\xa3", "\xf0\x9d\x97\xa4", "\xf0\x9d\x97\xa5"  /* f8 */
};

static const char *upcase1d6[] = {
  "\xf0\x9d\x97\xa6", "\xf0\x9d\x97\xa7", "\xf0\x9d\x97\xa8", "\xf0\x9d\x97\xa9", "\xf0\x9d\x97\xaa", "\xf0\x9d\x97\xab", "\xf0\x9d\x97\xac", "\xf0\x9d\x97\xad", /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 18 */
  NULL, NULL, "\xf0\x9d\x98\x88", "\xf0\x9d\x98\x89", "\xf0\x9d\x98\x8a", "\xf0\x9d\x98\x8b", "\xf0\x9d\x98\x8c", "\xf0\x9d\x98\x8d",                             /* 20 */
  "\xf0\x9d\x98\x8e", "\xf0\x9d\x98\x8f", "\xf0\x9d\x98\x90", "\xf0\x9d\x98\x91", "\xf0\x9d\x98\x92", "\xf0\x9d\x98\x93", "\xf0\x9d\x98\x94", "\xf0\x9d\x98\x95", /* 28 */
  "\xf0\x9d\x98\x96", "\xf0\x9d\x98\x97", "\xf0\x9d\x98\x98", "\xf0\x9d\x98\x99", "\xf0\x9d\x98\x9a", "\xf0\x9d\x98\x9b", "\xf0\x9d\x98\x9c", "\xf0\x9d\x98\x9d", /* 30 */
  "\xf0\x9d\x98\x9e", "\xf0\x9d\x98\x9f", "\xf0\x9d\x98\xa0", "\xf0\x9d\x98\xa1", NULL, NULL, NULL, NULL,                                                         /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xf0\x9d\x98\xbc", "\xf0\x9d\x98\xbd",                                                                                     /* 50 */
  "\xf0\x9d\x98\xbe", "\xf0\x9d\x98\xbf", "\xf0\x9d\x99\x80", "\xf0\x9d\x99\x81", "\xf0\x9d\x99\x82", "\xf0\x9d\x99\x83", "\xf0\x9d\x99\x84", "\xf0\x9d\x99\x85", /* 58 */
  "\xf0\x9d\x99\x86", "\xf0\x9d\x99\x87", "\xf0\x9d\x99\x88", "\xf0\x9d\x99\x89", "\xf0\x9d\x99\x8a", "\xf0\x9d\x99\x8b", "\xf0\x9d\x99\x8c", "\xf0\x9d\x99\x8d", /* 60 */
  "\xf0\x9d\x99\x8e", "\xf0\x9d\x99\x8f", "\xf0\x9d\x99\x90", "\xf0\x9d\x99\x91", "\xf0\x9d\x99\x92", "\xf0\x9d\x99\x93", "\xf0\x9d\x99\x94", "\xf0\x9d\x99\x95", /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 80 */
  NULL, NULL, "\xf0\x9d\x99\xb0", "\xf0\x9d\x99\xb1", "\xf0\x9d\x99\xb2", "\xf0\x9d\x99\xb3", "\xf0\x9d\x99\xb4", "\xf0\x9d\x99\xb5",                             /* 88 */
  "\xf0\x9d\x99\xb6", "\xf0\x9d\x99\xb7", "\xf0\x9d\x99\xb8", "\xf0\x9d\x99\xb9", "\xf0\x9d\x99\xba", "\xf0\x9d\x99\xbb", "\xf0\x9d\x99\xbc", "\xf0\x9d\x99\xbd", /* 90 */
  "\xf0\x9d\x99\xbe", "\xf0\x9d\x99\xbf", "\xf0\x9d\x9a\x80", "\xf0\x9d\x9a\x81", "\xf0\x9d\x9a\x82", "\xf0\x9d\x9a\x83", "\xf0\x9d\x9a\x84", "\xf0\x9d\x9a\x85", /* 98 */
  "\xf0\x9d\x9a\x86", "\xf0\x9d\x9a\x87", "\xf0\x9d\x9a\x88", "\xf0\x9d\x9a\x89", NULL, NULL, NULL, NULL,                                                         /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                                                                                                                  /* f8 */
};

static const char *upcasee00[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, "\xf3\xa0\x81\x81", "\xf3\xa0\x81\x82", "\xf3\xa0\x81\x83", "\xf3\xa0\x81\x84", "\xf3\xa0\x81\x85", "\xf3\xa0\x81\x86", "\xf3\xa0\x81\x87",               /* 60 */
  "\xf3\xa0\x81\x88", "\xf3\xa0\x81\x89", "\xf3\xa0\x81\x8a", "\xf3\xa0\x81\x8b", "\xf3\xa0\x81\x8c", "\xf3\xa0\x81\x8d", "\xf3\xa0\x81\x8e", "\xf3\xa0\x81\x8f", /* 68 */
  "\xf3\xa0\x81\x90", "\xf3\xa0\x81\x91", "\xf3\xa0\x81\x92", "\xf3\xa0\x81\x93", "\xf3\xa0\x81\x94", "\xf3\xa0\x81\x95", "\xf3\xa0\x81\x96", "\xf3\xa0\x81\x97", /* 70 */
  "\xf3\xa0\x81\x98", "\xf3\xa0\x81\x99", "\xf3\xa0\x81\x9a", NULL, NULL, NULL, NULL, NULL,                                                                       /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};


/* +++ Lowercase-Tabellen +++ */

static const char *lwcase000[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, "a", "b", "c", "d", "e", "f", "g",                         /* 40 */
  "h", "i", "j", "k", "l", "m", "n", "o",                          /* 48 */
  "p", "q", "r", "s", "t", "u", "v", "w",                          /* 50 */
  "x", "y", "z", NULL, NULL, NULL, NULL, NULL,                     /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  "\xc3\xa0", "\xc3\xa1", "\xc3\xa2", "\xc3\xa3", "\xc3\xa4", "\xc3\xa5", "\xc3\xa6", "\xc3\xa7",  /* c0 */
  "\xc3\xa8", "\xc3\xa9", "\xc3\xaa", "\xc3\xab", "\xc3\xac", "\xc3\xad", "\xc3\xae", "\xc3\xaf",  /* c8 */
  "\xc3\xb0", "\xc3\xb1", "\xc3\xb2", "\xc3\xb3", "\xc3\xb4", "\xc3\xb5", "\xc3\xb6", NULL,        /* d0 */
  "\xc3\xb8", "\xc3\xb9", "\xc3\xba", "\xc3\xbb", "\xc3\xbc", "\xc3\xbd", "\xc3\xbe", NULL,        /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *lwcase000_iso[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, "a", "b", "c", "d", "e", "f", "g",                         /* 40 */
  "h", "i", "j", "k", "l", "m", "n", "o",                          /* 48 */
  "p", "q", "r", "s", "t", "u", "v", "w",                          /* 50 */
  "x", "y", "z", NULL, NULL, NULL, NULL, NULL,                     /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  "\xe0", "\xe1", "\xe2", "\xe3", "\xe4", "\xe5", "\xe6", "\xe7",  /* c0 */
  "\xe8", "\xe9", "\xea", "\xeb", "\xec", "\xed", "\xee", "\xef",  /* c8 */
  "\xf0", "\xf1", "\xf2", "\xf3", "\xf4", "\xf5", "\xf6", NULL,    /* d0 */
  "\xf8", "\xf9", "\xfa", "\xfb", "\xfc", "\xfd", "\xfe", NULL,    /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *lwcase001[] = {
  "\xc4\x81", NULL, "\xc4\x83", NULL, "\xc4\x85", NULL, "\xc4\x87", NULL,      /* 00 */
  "\xc4\x89", NULL, "\xc4\x8b", NULL, "\xc4\x8d", NULL, "\xc4\x8f", NULL,      /* 08 */
  "\xc4\x91", NULL, "\xc4\x93", NULL, "\xc4\x95", NULL, "\xc4\x97", NULL,      /* 10 */
  "\xc4\x99", NULL, "\xc4\x9b", NULL, "\xc4\x9d", NULL, "\xc4\x9f", NULL,      /* 18 */
  "\xc4\xa1", NULL, "\xc4\xa3", NULL, "\xc4\xa5", NULL, "\xc4\xa7", NULL,      /* 20 */
  "\xc4\xa9", NULL, "\xc4\xab", NULL, "\xc4\xad", NULL, "\xc4\xaf", NULL,      /* 28 */
  NULL, NULL, "\xc4\xb3", NULL, "\xc4\xb5", NULL, "\xc4\xb7", NULL,            /* 30 */
  NULL, "\xc4\xba", NULL, "\xc4\xbc", NULL, "\xc4\xbe", NULL, "\xc5\x80",      /* 38 */
  NULL, "\xc5\x82", NULL, "\xc5\x84", NULL, "\xc5\x86", NULL, "\xc5\x88",      /* 40 */
  NULL, NULL, "\xc5\x8b", NULL, "\xc5\x8d", NULL, "\xc5\x8f", NULL,            /* 48 */
  "\xc5\x91", NULL, "\xc5\x93", NULL, "\xc5\x95", NULL, "\xc5\x97", NULL,      /* 50 */
  "\xc5\x99", NULL, "\xc5\x9b", NULL, "\xc5\x9d", NULL, "\xc5\x9f", NULL,      /* 58 */
  "\xc5\xa1", NULL, "\xc5\xa3", NULL, "\xc5\xa5", NULL, "\xc5\xa7", NULL,      /* 60 */
  "\xc5\xa9", NULL, "\xc5\xab", NULL, "\xc5\xad", NULL, "\xc5\xaf", NULL,      /* 68 */
  "\xc5\xb1", NULL, "\xc5\xb3", NULL, "\xc5\xb5", NULL, "\xc5\xb7", NULL,      /* 70 */
  "\xc3\xbf", "\xc5\xba", NULL, "\xc5\xbc", NULL, "\xc5\xbe", NULL, NULL,      /* 78 */
  NULL, NULL, "\xc6\x83", NULL, "\xc6\x85", NULL, NULL, "\xc6\x88",            /* 80 */
  NULL, NULL, NULL, "\xc6\x8c", NULL, NULL, NULL, NULL,                        /* 88 */
  NULL, "\xc6\x92", NULL, NULL, NULL, NULL, NULL, NULL,                        /* 90 */
  "\xc6\x99", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                        /* 98 */
  "\xc6\xa1", NULL, "\xc6\xa3", NULL, "\xc6\xa5", NULL, NULL, "\xc6\xa8",      /* a0 */
  NULL, NULL, NULL, NULL, "\xc6\xad", NULL, NULL, "\xc6\xb0",                  /* a8 */
  NULL, NULL, NULL, "\xc6\xb4", NULL, "\xc6\xb6", NULL, NULL,                  /* b0 */
  "\xc6\xb9", NULL, NULL, NULL, "\xc6\xbd", NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, "\xc7\x86", NULL, NULL, "\xc7\x89",                  /* c0 */
  NULL, NULL, "\xc7\x8c", NULL, NULL, "\xc7\x8e", NULL, "\xc7\x90",            /* c8 */
  NULL, "\xc7\x92", NULL, "\xc7\x94", NULL, "\xc7\x96", NULL, "\xc7\x98",      /* d0 */
  NULL, "\xc7\x9a", NULL, "\xc7\x9c", NULL, NULL, "\xc7\x9f", NULL,            /* d8 */
  "\xc7\xa1", NULL, "\xc7\xa3", NULL, "\xc7\xa5", NULL, "\xc7\xa7", NULL,      /* e0 */
  "\xc7\xa9", NULL, "\xc7\xab", NULL, "\xc7\xad", NULL, "\xc7\xaf", NULL,      /* e8 */
  NULL, "\xc7\xb3", NULL, NULL, "\xc7\xb5", NULL, NULL, NULL,                  /* f0 */
  "\xc7\xb9", NULL, "\xc7\xbb", NULL, "\xc7\xbd", NULL, "\xc7\xbf", NULL       /* f8 */
};

static const char *lwcase002[] = {
  "\xc8\x81", NULL, "\xc8\x83", NULL, "\xc8\x85", NULL, "\xc8\x87", NULL,      /* 00 */
  "\xc8\x89", NULL, "\xc8\x8b", NULL, "\xc8\x8d", NULL, "\xc8\x8f", NULL,      /* 08 */
  "\xc8\x91", NULL, "\xc8\x93", NULL, "\xc8\x95", NULL, "\xc8\x97", NULL,      /* 10 */
  "\xc8\x99", NULL, "\xc8\x9b", NULL, "\xc8\x9d", NULL, "\xc8\x9f", NULL,      /* 18 */
  NULL, NULL, "\xc8\xa3", NULL, "\xc8\xa5", NULL, "\xc8\xa7", NULL,            /* 20 */
  "\xc8\xa9", NULL, "\xc8\xab", NULL, "\xc8\xad", NULL, "\xc8\xaf", NULL,      /* 28 */
  "\xc8\xb1", NULL, "\xc8\xb3", NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, "\xc8\xbc", NULL, NULL, NULL, NULL,                        /* 38 */
  NULL, "\xc9\x82", NULL, NULL, NULL, NULL, "\xc9\x87", NULL,                  /* 40 */
  "\xc9\x89", NULL, "\xc9\x8b", NULL, "\xc9\x8d", NULL, "\xc9\x8f", NULL,      /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                              /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                               /* f8 */
};

static const char *lwcase01e[] = {
  "\xe1\xb8\x81", NULL, "\xe1\xb8\x83", NULL, "\xe1\xb8\x85", NULL, "\xe1\xb8\x87", NULL,      /* 00 */
  "\xe1\xb8\x89", NULL, "\xe1\xb8\x8b", NULL, "\xe1\xb8\x8d", NULL, "\xe1\xb8\x8f", NULL,      /* 08 */
  "\xe1\xb8\x91", NULL, "\xe1\xb8\x93", NULL, "\xe1\xb8\x95", NULL, "\xe1\xb8\x97", NULL,      /* 10 */
  "\xe1\xb8\x99", NULL, "\xe1\xb8\x9b", NULL, "\xe1\xb8\x9d", NULL, "\xe1\xb8\x9f", NULL,      /* 18 */
  "\xe1\xb8\xa1", NULL, "\xe1\xb8\xa3", NULL, "\xe1\xb8\xa5", NULL, "\xe1\xb8\xa7", NULL,      /* 20 */
  "\xe1\xb8\xa9", NULL, "\xe1\xb8\xab", NULL, "\xe1\xb8\xad", NULL, "\xe1\xb8\xaf", NULL,      /* 28 */
  "\xe1\xb8\xb1", NULL, "\xe1\xb8\xb3", NULL, "\xe1\xb8\xb5", NULL, "\xe1\xb8\xb7", NULL,      /* 30 */
  "\xe1\xb8\xb9", NULL, "\xe1\xb8\xbb", NULL, "\xe1\xb8\xbd", NULL, "\xe1\xb8\xbf", NULL,      /* 38 */
  "\xe1\xb9\x81", NULL, "\xe1\xb9\x83", NULL, "\xe1\xb9\x85", NULL, "\xe1\xb9\x87", NULL,      /* 40 */
  "\xe1\xb9\x89", NULL, "\xe1\xb9\x8b", NULL, "\xe1\xb9\x8d", NULL, "\xe1\xb9\x8f", NULL,      /* 48 */
  "\xe1\xb9\x91", NULL, "\xe1\xb9\x93", NULL, "\xe1\xb9\x95", NULL, "\xe1\xb9\x97", NULL,      /* 50 */
  "\xe1\xb9\x99", NULL, "\xe1\xb9\x9b", NULL, "\xe1\xb9\x9d", NULL, "\xe1\xb9\x9f", NULL,      /* 58 */
  "\xe1\xb9\xa1", NULL, "\xe1\xb9\xa3", NULL, "\xe1\xb9\xa5", NULL, "\xe1\xb9\xa7", NULL,      /* 60 */
  "\xe1\xb9\xa9", NULL, "\xe1\xb9\xab", NULL, "\xe1\xb9\xad", NULL, "\xe1\xb9\xaf", NULL,      /* 68 */
  "\xe1\xb9\xb1", NULL, "\xe1\xb9\xb3", NULL, "\xe1\xb9\xb5", NULL, "\xe1\xb9\xb7", NULL,      /* 70 */
  "\xe1\xb9\xb9", NULL, "\xe1\xb9\xbb", NULL, "\xe1\xb9\xbd", NULL, "\xe1\xb9\xbf", NULL,      /* 78 */
  "\xe1\xba\x81", NULL, "\xe1\xba\x83", NULL, "\xe1\xba\x85", NULL, "\xe1\xba\x87", NULL,      /* 80 */
  "\xe1\xba\x89", NULL, "\xe1\xba\x8b", NULL, "\xe1\xba\x8d", NULL, "\xe1\xba\x8f", NULL,      /* 88 */
  "\xe1\xba\x91", NULL, "\xe1\xba\x93", NULL, "\xe1\xba\x95", NULL, NULL, NULL,                /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, "\xc3\x9f", NULL,                                        /* 98 */
  "\xe1\xba\xa1", NULL, "\xe1\xba\xa3", NULL, "\xe1\xba\xa5", NULL, "\xe1\xba\xa7", NULL,      /* a0 */
  "\xe1\xba\xa9", NULL, "\xe1\xba\xab", NULL, "\xe1\xba\xad", NULL, "\xe1\xba\xaf", NULL,      /* a8 */
  "\xe1\xba\xb1", NULL, "\xe1\xba\xb3", NULL, "\xe1\xba\xb5", NULL, "\xe1\xba\xb7", NULL,      /* b0 */
  "\xe1\xba\xb9", NULL, "\xe1\xba\xbb", NULL, "\xe1\xba\xbd", NULL, "\xe1\xba\xbf", NULL,      /* b8 */
  "\xe1\xbb\x81", NULL, "\xe1\xbb\x83", NULL, "\xe1\xbb\x85", NULL, "\xe1\xbb\x87", NULL,      /* c0 */
  "\xe1\xbb\x89", NULL, "\xe1\xbb\x8b", NULL, "\xe1\xbb\x8d", NULL, "\xe1\xbb\x8f", NULL,      /* c8 */
  "\xe1\xbb\x91", NULL, "\xe1\xbb\x93", NULL, "\xe1\xbb\x95", NULL, "\xe1\xbb\x97", NULL,      /* d0 */
  "\xe1\xbb\x99", NULL, "\xe1\xbb\x9b", NULL, "\xe1\xbb\x9d", NULL, "\xe1\xbb\x9f", NULL,      /* d8 */
  "\xe1\xbb\xa1", NULL, "\xe1\xbb\xa3", NULL, "\xe1\xbb\xa5", NULL, "\xe1\xbb\xa7", NULL,      /* e0 */
  "\xe1\xbb\xa9", NULL, "\xe1\xbb\xab", NULL, "\xe1\xbb\xad", NULL, "\xe1\xbb\xaf", NULL,      /* e8 */
  "\xe1\xbb\xb1", NULL, "\xe1\xbb\xb3", NULL, "\xe1\xbb\xb5", NULL, "\xe1\xbb\xb7", NULL,      /* f0 */
  "\xe1\xbb\xb9", NULL, "\xe1\xbb\xbb", NULL, "\xe1\xbb\xbd", NULL, "\xe1\xbb\xbf", NULL       /* f8 */
};

static const char *lwcase02c[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 58 */
  "\xe2\xb1\xa1", NULL, NULL, NULL, NULL, NULL, NULL, "\xe2\xb1\xa8",            /* 60 */
  NULL, "\xe2\xb1\xaa", NULL, "\xe2\xb1\xac", NULL, NULL, NULL, NULL,            /* 68 */
  NULL, NULL, "\xe2\xb1\xb3", NULL, NULL, NULL, NULL, NULL,                      /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                                 /* f8 */
};

static const char *lwcase0a7[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 18 */
  NULL, NULL, "\xea\x9c\xa3", NULL, "\xea\x9c\xa5", NULL, "\xea\x9c\xa7", NULL,            /* 20 */
  "\xea\x9c\xa9", NULL, "\xea\x9c\xab", NULL, "\xea\x9c\xad", NULL, "\xea\x9c\xaf", NULL,  /* 28 */
  NULL, NULL, "\xea\x9c\xb3", NULL, "\xea\x9c\xb5", NULL, "\xea\x9c\xb7", NULL,            /* 30 */
  "\xea\x9c\xb9", NULL, "\xea\x9c\xbb", NULL, "\xea\x9c\xbd", NULL, "\xea\x9c\xbf", NULL,  /* 38 */
  "\xea\x9d\x81", NULL, "\xea\x9d\x83", NULL, "\xea\x9d\x85", NULL, "\xea\x9d\x87", NULL,  /* 40 */
  "\xea\x9d\x89", NULL, "\xea\x9d\x8b", NULL, "\xea\x9d\x8d", NULL, "\xea\x9d\x8f", NULL,  /* 48 */
  "\xea\x9d\x91", NULL, "\xea\x9d\x93", NULL, "\xea\x9d\x95", NULL, "\xea\x9d\x97", NULL,  /* 50 */
  "\xea\x9d\x99", NULL, "\xea\x9d\x9b", NULL, "\xea\x9d\x9d", NULL, "\xea\x9d\x9f", NULL,  /* 58 */
  "\xea\x9d\xa1", NULL, "\xea\x9d\xa3", NULL, "\xea\x9d\xa5", NULL, "\xea\x9d\xa7", NULL,  /* 60 */
  "\xea\x9d\xa9", NULL, "\xea\x9d\xab", NULL, "\xea\x9d\xad", NULL, "\xea\x9d\xaf", NULL,  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 70 */
  NULL, "\xea\x9d\xba", NULL, "\xea\x9d\xbc", NULL, NULL, "\xea\x9d\xbf", NULL,            /* 78 */
  "\xea\x9e\x81", NULL, "\xea\x9e\x83", NULL, "\xea\x9e\x85", NULL, "\xea\x9e\x87", NULL,  /* 80 */
  NULL, NULL, NULL, "\xea\x9e\x8c", NULL, NULL, NULL, NULL,                                /* 88 */
  "\xea\x9e\x91", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* 98 */
  "\xea\x9e\xa1", NULL, "\xea\x9e\xa3", NULL, "\xea\x9e\xa5", NULL, "\xea\x9e\xa7", NULL,  /* a0 */
  "\xea\x9e\xa9", NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                          /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                                           /* f8 */
};

static const char *lwcase0ff[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, "\xef\xbd\x81", "\xef\xbd\x82", "\xef\xbd\x83", "\xef\xbd\x84", "\xef\xbd\x85", "\xef\xbd\x86", "\xef\xbd\x87",            /* 20 */
  "\xef\xbd\x88", "\xef\xbd\x89", "\xef\xbd\x8a", "\xef\xbd\x8b", "\xef\xbd\x8c", "\xef\xbd\x8d", "\xef\xbd\x8e", "\xef\xbd\x8f",  /* 28 */
  "\xef\xbd\x90", "\xef\xbd\x91", "\xef\xbd\x92", "\xef\xbd\x93", "\xef\xbd\x94", "\xef\xbd\x95", "\xef\xbd\x96", "\xef\xbd\x97",  /* 30 */
  "\xef\xbd\x98", "\xef\xbd\x99", "\xef\xbd\x9a", NULL, NULL, NULL, NULL, NULL,                                                    /* 38 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 40 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};

static const char *lwcase1d4[] = {
  "\xf0\x9d\x90\x9a", "\xf0\x9d\x90\x9b", "\xf0\x9d\x90\x9c", "\xf0\x9d\x90\x9d", "\xf0\x9d\x90\x9e", "\xf0\x9d\x90\x9f", "\xf0\x9d\x90\xa0", "\xf0\x9d\x90\xa1", /* 00 */
  "\xf0\x9d\x90\xa2", "\xf0\x9d\x90\xa3", "\xf0\x9d\x90\xa4", "\xf0\x9d\x90\xa5", "\xf0\x9d\x90\xa6", "\xf0\x9d\x90\xa7", "\xf0\x9d\x90\xa8", "\xf0\x9d\x90\xa9", /* 08 */
  "\xf0\x9d\x90\xaa", "\xf0\x9d\x90\xab", "\xf0\x9d\x90\xac", "\xf0\x9d\x90\xad", "\xf0\x9d\x90\xae", "\xf0\x9d\x90\xaf", "\xf0\x9d\x90\xb0", "\xf0\x9d\x90\xb1", /* 10 */
  "\xf0\x9d\x90\xb2", "\xf0\x9d\x90\xb3", NULL, NULL, NULL, NULL, NULL, NULL,                                                                                     /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 28 */
  NULL, NULL, NULL, NULL, "\xf0\x9d\x91\x8e", "\xf0\x9d\x91\x8f", "\xf0\x9d\x91\x90", "\xf0\x9d\x91\x91",                                                         /* 30 */
  "\xf0\x9d\x91\x92", "\xf0\x9d\x91\x93", "\xf0\x9d\x91\x94", "\xf0\x9d\x91\x95", "\xf0\x9d\x91\x96", "\xf0\x9d\x91\x97", "\xf0\x9d\x91\x98", "\xf0\x9d\x91\x99", /* 38 */
  "\xf0\x9d\x91\x9a", "\xf0\x9d\x91\x9b", "\xf0\x9d\x91\x9c", "\xf0\x9d\x91\x9d", "\xf0\x9d\x91\x9e", "\xf0\x9d\x91\x9f", "\xf0\x9d\x91\xa0", "\xf0\x9d\x91\xa1", /* 40 */
  "\xf0\x9d\x91\xa2", "\xf0\x9d\x91\xa3", "\xf0\x9d\x91\xa4", "\xf0\x9d\x91\xa5", "\xf0\x9d\x91\xa6", "\xf0\x9d\x91\xa7", NULL, NULL,                             /* 48 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 60 */
  "\xf0\x9d\x92\x82", "\xf0\x9d\x92\x83", "\xf0\x9d\x92\x84", "\xf0\x9d\x92\x85", "\xf0\x9d\x92\x86", "\xf0\x9d\x92\x87", "\xf0\x9d\x92\x88", "\xf0\x9d\x92\x89", /* 68 */
  "\xf0\x9d\x92\x8a", "\xf0\x9d\x92\x8b", "\xf0\x9d\x92\x8c", "\xf0\x9d\x92\x8d", "\xf0\x9d\x92\x8e", "\xf0\x9d\x92\x8f", "\xf0\x9d\x92\x90", "\xf0\x9d\x92\x91", /* 70 */
  "\xf0\x9d\x92\x92", "\xf0\x9d\x92\x93", "\xf0\x9d\x92\x94", "\xf0\x9d\x92\x95", "\xf0\x9d\x92\x96", "\xf0\x9d\x92\x97", "\xf0\x9d\x92\x98", "\xf0\x9d\x92\x99", /* 78 */
  "\xf0\x9d\x92\x9a", "\xf0\x9d\x92\x9b", NULL, NULL, NULL, NULL, NULL, NULL,                                                                                     /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 90 */
  NULL, NULL, NULL, NULL, "\xf0\x9d\x92\xb6", "\xf0\x9d\x92\xb7", "\xf0\x9d\x92\xb8", "\xf0\x9d\x92\xb9",                                                         /* 98 */
  "\xf0\x9d\x92\xba", "\xf0\x9d\x92\xbb", "\xf0\x9d\x92\xbc", "\xf0\x9d\x92\xbd", "\xf0\x9d\x92\xbe", "\xf0\x9d\x92\xbf", "\xf0\x9d\x93\x80", "\xf0\x9d\x93\x81", /* a0 */
  "\xf0\x9d\x93\x82", "\xf0\x9d\x93\x83", "\xf0\x9d\x93\x84", "\xf0\x9d\x93\x85", "\xf0\x9d\x93\x86", "\xf0\x9d\x93\x87", "\xf0\x9d\x93\x88", "\xf0\x9d\x93\x89", /* a8 */
  "\xf0\x9d\x93\x8a", "\xf0\x9d\x93\x8b", "\xf0\x9d\x93\x8c", "\xf0\x9d\x93\x8d", "\xf0\x9d\x93\x8e", "\xf0\x9d\x93\x8f", NULL, NULL,                             /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* c8 */
  "\xf0\x9d\x93\xaa", "\xf0\x9d\x93\xab", "\xf0\x9d\x93\xac", "\xf0\x9d\x93\xad", "\xf0\x9d\x93\xae", "\xf0\x9d\x93\xaf", "\xf0\x9d\x93\xb0", "\xf0\x9d\x93\xb1", /* d0 */
  "\xf0\x9d\x93\xb2", "\xf0\x9d\x93\xb3", "\xf0\x9d\x93\xb4", "\xf0\x9d\x93\xb5", "\xf0\x9d\x93\xb6", "\xf0\x9d\x93\xb7", "\xf0\x9d\x93\xb8", "\xf0\x9d\x93\xb9", /* d8 */
  "\xf0\x9d\x93\xba", "\xf0\x9d\x93\xbb", "\xf0\x9d\x93\xbc", "\xf0\x9d\x93\xbd", "\xf0\x9d\x93\xbe", "\xf0\x9d\x93\xbf", "\xf0\x9d\x94\x80", "\xf0\x9d\x94\x81", /* e0 */
  "\xf0\x9d\x94\x82", "\xf0\x9d\x94\x83", NULL, NULL, NULL, NULL, NULL, NULL,                                                                                     /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                                                                                                                  /* f8 */
};

static const char *lwcase1d5[] = {
  NULL, NULL, NULL, NULL, "\xf0\x9d\x94\x9e", "\xf0\x9d\x94\x9f", "\xf0\x9d\x94\xa0", "\xf0\x9d\x94\xa1",                                                         /* 00 */
  "\xf0\x9d\x94\xa2", "\xf0\x9d\x94\xa3", "\xf0\x9d\x94\xa4", "\xf0\x9d\x94\xa5", "\xf0\x9d\x94\xa6", "\xf0\x9d\x94\xa7", "\xf0\x9d\x94\xa8", "\xf0\x9d\x94\xa9", /* 08 */
  "\xf0\x9d\x94\xaa", "\xf0\x9d\x94\xab", "\xf0\x9d\x94\xac", "\xf0\x9d\x94\xad", "\xf0\x9d\x94\xae", "\xf0\x9d\x94\xaf", "\xf0\x9d\x94\xb0", "\xf0\x9d\x94\xb1", /* 10 */
  "\xf0\x9d\x94\xb2", "\xf0\x9d\x94\xb3", "\xf0\x9d\x94\xb4", "\xf0\x9d\x94\xb5", "\xf0\x9d\x94\xb6", "\xf0\x9d\x94\xb7", NULL, NULL,                             /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 30 */
  "\xf0\x9d\x95\x92", "\xf0\x9d\x95\x93", "\xf0\x9d\x95\x94", "\xf0\x9d\x95\x95", "\xf0\x9d\x95\x96", "\xf0\x9d\x95\x97", "\xf0\x9d\x95\x98", "\xf0\x9d\x95\x99", /* 38 */
  "\xf0\x9d\x95\x9a", "\xf0\x9d\x95\x9b", "\xf0\x9d\x95\x9c", "\xf0\x9d\x95\x9d", "\xf0\x9d\x95\x9e", "\xf0\x9d\x95\x9f", "\xf0\x9d\x95\xa0", "\xf0\x9d\x95\xa1", /* 40 */
  "\xf0\x9d\x95\xa2", "\xf0\x9d\x95\xa3", "\xf0\x9d\x95\xa4", "\xf0\x9d\x95\xa5", "\xf0\x9d\x95\xa6", "\xf0\x9d\x95\xa7", "\xf0\x9d\x95\xa8", "\xf0\x9d\x95\xa9", /* 48 */
  "\xf0\x9d\x95\xaa", "\xf0\x9d\x95\xab", NULL, NULL, NULL, NULL, NULL, NULL,                                                                                     /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 60 */
  NULL, NULL, NULL, NULL, "\xf0\x9d\x96\x86", "\xf0\x9d\x96\x87", "\xf0\x9d\x96\x88", "\xf0\x9d\x96\x89",                                                         /* 68 */
  "\xf0\x9d\x96\x8a", "\xf0\x9d\x96\x8b", "\xf0\x9d\x96\x8c", "\xf0\x9d\x96\x8d", "\xf0\x9d\x96\x8e", "\xf0\x9d\x96\x8f", "\xf0\x9d\x96\x90", "\xf0\x9d\x96\x91", /* 70 */
  "\xf0\x9d\x96\x92", "\xf0\x9d\x96\x93", "\xf0\x9d\x96\x94", "\xf0\x9d\x96\x95", "\xf0\x9d\x96\x96", "\xf0\x9d\x96\x97", "\xf0\x9d\x96\x98", "\xf0\x9d\x96\x99", /* 78 */
  "\xf0\x9d\x96\x9a", "\xf0\x9d\x96\x9b", "\xf0\x9d\x96\x9c", "\xf0\x9d\x96\x9d", "\xf0\x9d\x96\x9e", "\xf0\x9d\x96\x9f", NULL, NULL,                             /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 98 */
  "\xf0\x9d\x96\xba", "\xf0\x9d\x96\xbb", "\xf0\x9d\x96\xbc", "\xf0\x9d\x96\xbd", "\xf0\x9d\x96\xbe", "\xf0\x9d\x96\xbf", "\xf0\x9d\x97\x80", "\xf0\x9d\x97\x81", /* a0 */
  "\xf0\x9d\x97\x82", "\xf0\x9d\x97\x83", "\xf0\x9d\x97\x84", "\xf0\x9d\x97\x85", "\xf0\x9d\x97\x86", "\xf0\x9d\x97\x87", "\xf0\x9d\x97\x88", "\xf0\x9d\x97\x89", /* a8 */
  "\xf0\x9d\x97\x8a", "\xf0\x9d\x97\x8b", "\xf0\x9d\x97\x8c", "\xf0\x9d\x97\x8d", "\xf0\x9d\x97\x8e", "\xf0\x9d\x97\x8f", "\xf0\x9d\x97\x90", "\xf0\x9d\x97\x91", /* b0 */
  "\xf0\x9d\x97\x91", "\xf0\x9d\x97\x92", NULL, NULL, NULL, NULL, NULL, NULL,                                                                                     /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* c8 */
  NULL, NULL, NULL, NULL, "\xf0\x9d\x97\xae", "\xf0\x9d\x97\xaf", "\xf0\x9d\x97\xb0", "\xf0\x9d\x97\xb1",                                                         /* d0 */
  "\xf0\x9d\x97\xb2", "\xf0\x9d\x97\xb3", "\xf0\x9d\x97\xb4", "\xf0\x9d\x97\xb5", "\xf0\x9d\x97\xb6", "\xf0\x9d\x97\xb7", "\xf0\x9d\x97\xb8", "\xf0\x9d\x97\xb9", /* d8 */
  "\xf0\x9d\x97\xba", "\xf0\x9d\x97\xbb", "\xf0\x9d\x97\xbc", "\xf0\x9d\x97\xbd", "\xf0\x9d\x97\xbe", "\xf0\x9d\x97\xbf", "\xf0\x9d\x98\x80", "\xf0\x9d\x98\x81", /* e0 */
  "\xf0\x9d\x98\x82", "\xf0\x9d\x98\x83", "\xf0\x9d\x98\x84", "\xf0\x9d\x98\x85", "\xf0\x9d\x98\x86", "\xf0\x9d\x98\x87", NULL, NULL,                             /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                                                                                                                  /* f8 */
};

static const char *lwcase1d6[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 00 */
  "\xf0\x9d\x98\xa2", "\xf0\x9d\x98\xa3", "\xf0\x9d\x98\xa4", "\xf0\x9d\x98\xa5", "\xf0\x9d\x98\xa6", "\xf0\x9d\x98\xa7", "\xf0\x9d\x98\xa8", "\xf0\x9d\x98\xa9", /* 08 */
  "\xf0\x9d\x98\xaa", "\xf0\x9d\x98\xab", "\xf0\x9d\x98\xac", "\xf0\x9d\x98\xad", "\xf0\x9d\x98\xae", "\xf0\x9d\x98\xaf", "\xf0\x9d\x98\xb0", "\xf0\x9d\x98\xb1", /* 10 */
  "\xf0\x9d\x98\xb2", "\xf0\x9d\x98\xb3", "\xf0\x9d\x98\xb4", "\xf0\x9d\x98\xb5", "\xf0\x9d\x98\xb6", "\xf0\x9d\x98\xb7", "\xf0\x9d\x98\xb8", "\xf0\x9d\x98\xb9", /* 18 */
  "\xf0\x9d\x98\xba", "\xf0\x9d\x98\xbb", NULL, NULL, NULL, NULL, NULL, NULL,                                                                                     /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 30 */
  NULL, NULL, NULL, NULL, "\xf0\x9d\x99\x96", "\xf0\x9d\x99\x97", "\xf0\x9d\x99\x98", "\xf0\x9d\x99\x99",                                                         /* 38 */
  "\xf0\x9d\x99\x9a", "\xf0\x9d\x99\x9b", "\xf0\x9d\x99\x9c", "\xf0\x9d\x99\x9d", "\xf0\x9d\x99\x9e", "\xf0\x9d\x99\x9f", "\xf0\x9d\x99\xa0", "\xf0\x9d\x99\xa1", /* 40 */
  "\xf0\x9d\x99\xa2", "\xf0\x9d\x99\xa3", "\xf0\x9d\x99\xa4", "\xf0\x9d\x99\xa5", "\xf0\x9d\x99\xa6", "\xf0\x9d\x99\xa7", "\xf0\x9d\x99\xa8", "\xf0\x9d\x99\xa9", /* 48 */
  "\xf0\x9d\x99\xaa", "\xf0\x9d\x99\xab", "\xf0\x9d\x99\xac", "\xf0\x9d\x99\xad", "\xf0\x9d\x99\xae", "\xf0\x9d\x99\xaf", NULL, NULL,                             /* 50 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 68 */
  "\xf0\x9d\x9a\x8a", "\xf0\x9d\x9a\x8b", "\xf0\x9d\x9a\x8c", "\xf0\x9d\x9a\x8d", "\xf0\x9d\x9a\x8e", "\xf0\x9d\x9a\x8f", "\xf0\x9d\x9a\x90", "\xf0\x9d\x9a\x91", /* 70 */
  "\xf0\x9d\x9a\x92", "\xf0\x9d\x9a\x93", "\xf0\x9d\x9a\x94", "\xf0\x9d\x9a\x95", "\xf0\x9d\x9a\x96", "\xf0\x9d\x9a\x97", "\xf0\x9d\x9a\x98", "\xf0\x9d\x9a\x99", /* 78 */
  "\xf0\x9d\x9a\x9a", "\xf0\x9d\x9a\x9b", "\xf0\x9d\x9a\x9c", "\xf0\x9d\x9a\x9d", "\xf0\x9d\x9a\x9e", "\xf0\x9d\x9a\x9f", "\xf0\x9d\x9a\xa0", "\xf0\x9d\x9a\xa1", /* 80 */
  "\xf0\x9d\x9a\xa2", "\xf0\x9d\x9a\xa3", NULL, NULL, NULL, NULL, NULL, NULL,                                                                                     /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                                                                                                                 /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                                                                                                                  /* f8 */
};

static const char *lwcasee00[] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 00 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 08 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 10 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 18 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 20 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 28 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 30 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 38 */
  NULL, "\xf3\xa0\x81\xa1", "\xf3\xa0\x81\xa2", "\xf3\xa0\x81\xa3", "\xf3\xa0\x81\xa4", "\xf3\xa0\x81\xa5", "\xf3\xa0\x81\xa6", "\xf3\xa0\x81\xa7",               /* 40 */
  "\xf3\xa0\x81\xa8", "\xf3\xa0\x81\xa9", "\xf3\xa0\x81\xaa", "\xf3\xa0\x81\xab", "\xf3\xa0\x81\xac", "\xf3\xa0\x81\xad", "\xf3\xa0\x81\xae", "\xf3\xa0\x81\xaf", /* 48 */
  "\xf3\xa0\x81\xb0", "\xf3\xa0\x81\xb1", "\xf3\xa0\x81\xb2", "\xf3\xa0\x81\xb3", "\xf3\xa0\x81\xb4", "\xf3\xa0\x81\xb5", "\xf3\xa0\x81\xb6", "\xf3\xa0\x81\xb7", /* 50 */
  "\xf3\xa0\x81\xb8", "\xf3\xa0\x81\xb9", "\xf3\xa0\x81\xba", NULL, NULL, NULL, NULL, NULL,                                                                       /* 58 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 60 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 68 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 70 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 78 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 80 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 88 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 90 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* 98 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* a8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* b8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* c8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* d8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* e8 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,                  /* f0 */
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL                   /* f8 */
};


/* +++ Funktionen +++ */

struct tabrets {
  char s[6];
};

int SML3_latin_isutf8(const char *, size_t);
int SML3_latin_index(int, int, char *, size_t *);
int SML3_latin_next(int, const char *, size_t, char *, size_t *);
size_t SML3_latin_convert(int, const char *, size_t *, struct SML3_gummi *, size_t);
char * SML3_latin_toupper(int, const char *);
char * SML3_latin_tolower(int, const char *);
int SML3_latin_strlen(const char *);
int SML3_latin_strcmp(const char *, const char *, int);

static const char * modif_uml(int, const char *, int *);
static const char * idx_tab(int, int, int *, struct tabrets *);
static int next_utf8(const char *, size_t, int *);


/* modif_uml [thread-sicher]:
 * modifiziert Umlaut-Ersatz
 * 1.Arg: IO-Flag
 *        - evtl. aus SML3_LATIN_MODIF_*
 * 2.Arg: Zeichenstring
 * 3.Arg: Adresse auf Anzahl Bytes in Zeichenstring
 * Rueckgabe: neuer Zeichenstring
 */
static const char *
modif_uml(int ioflag, const char *rzch, int *ranz)
{
  if (*ranz != 2 || strchr("AOUaou", (int)rzch[0]) == NULL || rzch[1] != '?') { return rzch; }

  if (ioflag & SML3_LATIN_MODIF_UML) {
    switch(rzch[0]) {
      case 'A':  rzch = "AE"; break;
      case 'O':  rzch = "OE"; break;
      case 'U':  rzch = "UE"; break;
      case 'a':  rzch = "ae"; break;
      case 'o':  rzch = "oe"; break;
      case 'u':  rzch = "ue"; break;
    }
  } else {
    switch(rzch[0]) {
      case 'A':  rzch = "A"; *ranz = 1; break;
      case 'O':  rzch = "O"; *ranz = 1; break;
      case 'U':  rzch = "U"; *ranz = 1; break;
      case 'a':  rzch = "a"; *ranz = 1; break;
      case 'o':  rzch = "o"; *ranz = 1; break;
      case 'u':  rzch = "u"; *ranz = 1; break;
    }
  }

  return rzch;
} /* Ende modif_uml */


/* idx_tab [thread-sicher]:
 * gibt Zeichenstring je nach Ausgabeflag zurueck
 * 1.Arg: Zeichenindex
 * 2.Arg: IO-Flag
 *        - evtl. aus SML3_LATIN_INPUT_*
 *        - eins aus SML3_LATIN_OUTPUT_*
 *        - evtl. aus SML3_LATIN_MODIF_*
 * 3.Arg: fuer Rueckgabe Anzahl Bytes in Rueckgabewert
 * 4.Arg: interner Puffer
 * Rueckgabe: Zeichenstring oder NULL
 */
static const char *
idx_tab(int index, int ioflag, int *ranz, struct tabrets *tbr)
{
#ifdef SML3_HAVE_PTHREAD
  static pthread_mutex_t neu_mtx = PTHREAD_MUTEX_INITIALIZER;
#endif
  static volatile int neu = 1;
  static const char **asc_tabs[4096];
  static const char **iso_tabs[4096];
  static const char **asort_tabs[4096];
  static const char **upcase_tabs[4096];
  static const char **lwcase_tabs[4096];
  const char **tabelle, *rzch;
  int idxdiv, idxmod;

  *ranz = 0;
  if (index < 0) { return NULL; }
  idxdiv = index / 256;
  idxmod = index % 256;

  if (neu) {
#ifdef SML3_HAVE_PTHREAD
    pthread_mutex_lock(&neu_mtx);
#endif
    if (neu) {
      memset(asc_tabs, 0, sizeof(asc_tabs));
      asc_tabs[0] = asc000;
      asc_tabs[1] = asc001;
      asc_tabs[2] = asc002;
      asc_tabs[0x1d] = asc01d;
      asc_tabs[0x1e] = asc01e;
      asc_tabs[0x20] = asc020;
      asc_tabs[0x21] = asc021;
      asc_tabs[0x22] = asc022;
      asc_tabs[0x23] = asc023;
      asc_tabs[0x25] = asc025;
      asc_tabs[0x2c] = asc02c;
      asc_tabs[0xa7] = asc0a7;
      asc_tabs[0xfb] = asc0fb;
      asc_tabs[0xff] = asc0ff;
      asc_tabs[0x1d4] = asc1d4;
      asc_tabs[0x1d5] = asc1d5;
      asc_tabs[0x1d6] = asc1d6;
      asc_tabs[0x1d7] = asc1d7;
      asc_tabs[0xe00] = asce00;
      memset(iso_tabs, 0, sizeof(iso_tabs));
      iso_tabs[0] = iso000;
      iso_tabs[1] = iso001;
      iso_tabs[2] = iso002;
      iso_tabs[0x1d] = iso01d;
      iso_tabs[0x1e] = iso01e;
      iso_tabs[0x20] = iso020;
      iso_tabs[0x21] = iso021;
      iso_tabs[0x22] = iso022;
      iso_tabs[0x23] = iso023;
      iso_tabs[0x25] = iso025;
      iso_tabs[0xa7] = iso0a7;
      iso_tabs[0xff] = iso0ff;
      iso_tabs[0x1d6] = iso1d6;
      iso_tabs[0x1d7] = iso1d7;
      memset(asort_tabs, 0, sizeof(asort_tabs));
      asort_tabs[0] = asort000;
      asort_tabs[2] = asort002;
      asort_tabs[0x20] = asort020;
      asort_tabs[0x21] = asort021;
      asort_tabs[0x23] = asort023;
      asort_tabs[0x2c] = asort02c;
      asort_tabs[0xff] = asort0ff;
      asort_tabs[0x1d5] = asort1d5;
      asort_tabs[0x1d6] = asort1d6;
      asort_tabs[0x1d7] = asort1d7;
      memset(upcase_tabs, 0, sizeof(upcase_tabs));
      upcase_tabs[0] = upcase000;
      upcase_tabs[1] = upcase001;
      upcase_tabs[2] = upcase002;
      upcase_tabs[0x1e] = upcase01e;
      upcase_tabs[0x2c] = upcase02c;
      upcase_tabs[0xa7] = upcase0a7;
      upcase_tabs[0xff] = upcase0ff;
      upcase_tabs[0x1d4] = upcase1d4;
      upcase_tabs[0x1d5] = upcase1d5;
      upcase_tabs[0x1d6] = upcase1d6;
      upcase_tabs[0xe00] = upcasee00;
      memset(lwcase_tabs, 0, sizeof(lwcase_tabs));
      lwcase_tabs[0] = lwcase000;
      lwcase_tabs[1] = lwcase001;
      lwcase_tabs[2] = lwcase002;
      lwcase_tabs[0x1e] = lwcase01e;
      lwcase_tabs[0x2c] = lwcase02c;
      lwcase_tabs[0xa7] = lwcase0a7;
      lwcase_tabs[0xff] = lwcase0ff;
      lwcase_tabs[0x1d4] = lwcase1d4;
      lwcase_tabs[0x1d5] = lwcase1d5;
      lwcase_tabs[0x1d6] = lwcase1d6;
      lwcase_tabs[0xe00] = lwcasee00;
      neu = 0;
    }
#ifdef SML3_HAVE_PTHREAD
    pthread_mutex_unlock(&neu_mtx);
#endif
  }

  rzch = NULL;

  if (ioflag & SML3_LATIN_OUTPUT_ASCII) {
    if (idxdiv >= 4096) { return NULL; }
    tabelle = asc_tabs[idxdiv];
    if (tabelle != NULL) { rzch = tabelle[idxmod]; }
    if (rzch != NULL) {
      *ranz = strlen(rzch);
      if (*ranz == 0) { *ranz = 1; }  /* 0er-Zeichen */
      rzch = modif_uml(ioflag, rzch, ranz);
    }

  } else if (ioflag & SML3_LATIN_OUTPUT_ISO) {
    if (idxdiv >= 4096) { return NULL; }
    tabelle = iso_tabs[idxdiv];
    if (tabelle != NULL) { rzch = tabelle[idxmod]; }
    if (rzch == NULL) {
      tabelle = asc_tabs[idxdiv];
      if (tabelle != NULL) { rzch = tabelle[idxmod]; }
    }
    if (rzch != NULL) {
      *ranz = strlen(rzch);
      if (*ranz == 0) { *ranz = 1; }  /* 0er-Zeichen */
      rzch = modif_uml(ioflag, rzch, ranz);
    }

  } else if (ioflag & SML3_LATIN_OUTPUT_SORT) {
    if (idxdiv >= 4096) { return NULL; }
    tabelle = asort_tabs[idxdiv];
    if (tabelle != NULL) { rzch = tabelle[idxmod]; }
    if (rzch != NULL) {
      *ranz = strlen(rzch);
      if (*ranz == 0) { *ranz = 1; }  /* 0er-Zeichen */
      rzch = modif_uml(ioflag, rzch, ranz);
    }

  } else if (ioflag & SML3_LATIN_OUTPUT_UPCASE) {
    if (idxdiv >= 4096) { return NULL; }
    if (ioflag & SML3_LATIN_INPUT_ISO) {
      if (idxdiv == 0) { rzch = upcase000_iso[idxmod]; }
    } else {
      tabelle = upcase_tabs[idxdiv];
      if (tabelle != NULL) { rzch = tabelle[idxmod]; }
    }
    if (rzch != NULL) {
      *ranz = strlen(rzch);
      if (*ranz == 0) { *ranz = 1; }  /* 0er-Zeichen */
    }

  } else if (ioflag & SML3_LATIN_OUTPUT_LWCASE) {
    if (idxdiv >= 4096) { return NULL; }
    if (ioflag & SML3_LATIN_INPUT_ISO) {
      if (idxdiv == 0) { rzch = lwcase000_iso[idxmod]; }
    } else {
      tabelle = lwcase_tabs[idxdiv];
      if (tabelle != NULL) { rzch = tabelle[idxmod]; }
    }
    if (rzch != NULL) {
      *ranz = strlen(rzch);
      if (*ranz == 0) { *ranz = 1; }  /* 0er-Zeichen */
    }

  } else if (ioflag & SML3_LATIN_OUTPUT_UTF8) {
    if (index < 0x80) {
      tbr->s[0] = (char)index;
      *ranz = 1;
    } else if (index < 0x800) {
      tbr->s[0] = ((index >> 6) & 0x1f) | 0xc0;
      tbr->s[1] = (index & 0x3f) | 0x80;
      *ranz = 2;
    } else if (index < 0x10000) {
      tbr->s[0] = ((index >> 12) & 0x0f) | 0xe0;
      tbr->s[1] = ((index >> 6) & 0x3f) | 0x80;
      tbr->s[2] = (index & 0x3f) | 0x80;
      *ranz = 3;
    } else if (index < 0x200000) {
      tbr->s[0] = ((index >> 18) & 0x07) | 0xf0;
      tbr->s[1] = ((index >> 12) & 0x3f) | 0x80;
      tbr->s[2] = ((index >> 6) & 0x3f) | 0x80;
      tbr->s[3] = (index & 0x3f) | 0x80;
      *ranz = 4;
    } else if (index < 0x4000000) {
      tbr->s[0] = ((index >> 24) & 0x03) | 0xf8;
      tbr->s[1] = ((index >> 18) & 0x3f) | 0x80;
      tbr->s[2] = ((index >> 12) & 0x3f) | 0x80;
      tbr->s[3] = ((index >> 6) & 0x3f) | 0x80;
      tbr->s[4] = (index & 0x3f) | 0x80;
      *ranz = 5;
    } else {
      tbr->s[0] = ((index >> 30) & 0x01) | 0xfc;
      tbr->s[1] = ((index >> 24) & 0x3f) | 0x80;
      tbr->s[2] = ((index >> 18) & 0x3f) | 0x80;
      tbr->s[3] = ((index >> 12) & 0x3f) | 0x80;
      tbr->s[4] = ((index >> 6) & 0x3f) | 0x80;
      tbr->s[5] = (index & 0x3f) | 0x80;
      *ranz = 6;
    }
    rzch = tbr->s;
  }

  return rzch;
} /* Ende idx_tab */


/* next_utf8 [reentrant]:
 * ermittelt naechstes Zeichen
 * 1.Arg: String
 * 2.Arg: Stringlaenge
 * 3.Arg: fuer Rueckgabe Index
 * Rueckgabe: wieviel Bytes Zeichen enthaelt
 *            oder 0  = Ende
 *            oder -1 = ungueltiges Byte bzw. Zeichensequenz nicht beendet
 */
static int
next_utf8(const char *string, size_t size, int *index)
{
  unsigned char *uc;
  int anzz;

  if (string == NULL || size == 0 || index == NULL) { return 0; }

  uc = (unsigned char *)string;
  if (uc[0] < 0x80) { *index = (int)uc[0]; return 1; }
  if (uc[0] < 0xc2) { return -1; }

  if (uc[0] < 0xe0) {
    anzz = 2;
    if (size < (size_t)anzz) { return -1; }
    if (uc[1] < 0x80 || uc[1] > 0xbf) { return -1; }
    *index = ((uc[0] & 0x1f) << 6) | (uc[1] & 0x3f);
    return anzz;
  }

  if (uc[0] < 0xf0) {
    anzz = 3;
    if (size < (size_t)anzz) { return -1; }
    if (uc[1] < 0x80 || uc[1] > 0xbf) { return -1; }
    if (uc[2] < 0x80 || uc[2] > 0xbf) { return -1; }
    *index = ((uc[0] & 0x0f) << 12) | ((uc[1] & 0x3f) << 6) | (uc[2] & 0x3f);
    return anzz;
  }

  if (uc[0] < 0xf8) {
    anzz = 4;
    if (size < (size_t)anzz) { return -1; }
    if (uc[1] < 0x80 || uc[1] > 0xbf) { return -1; }
    if (uc[2] < 0x80 || uc[2] > 0xbf) { return -1; }
    if (uc[3] < 0x80 || uc[3] > 0xbf) { return -1; }
    *index = ((uc[0] & 0x07) << 18) | ((uc[1] & 0x3f) << 12) | ((uc[2] & 0x3f) << 6) | (uc[3] & 0x3f);
    return anzz;
  }

  if (uc[0] < 0xfc) {
    anzz = 5;
    if (size < (size_t)anzz) { return -1; }
    if (uc[1] < 0x80 || uc[1] > 0xbf) { return -1; }
    if (uc[2] < 0x80 || uc[2] > 0xbf) { return -1; }
    if (uc[3] < 0x80 || uc[3] > 0xbf) { return -1; }
    if (uc[4] < 0x80 || uc[4] > 0xbf) { return -1; }
    *index = ((uc[0] & 0x03) << 24) | ((uc[1] & 0x3f) << 18) | ((uc[2] & 0x3f) << 12) | ((uc[3] & 0x3f) << 6) | (uc[4] & 0x3f);
    return anzz;
  }

  if (uc[0] < 0xfe) {
    anzz = 6;
    if (size < (size_t)anzz) { return -1; }
    if (uc[1] < 0x80 || uc[1] > 0xbf) { return -1; }
    if (uc[2] < 0x80 || uc[2] > 0xbf) { return -1; }
    if (uc[3] < 0x80 || uc[3] > 0xbf) { return -1; }
    if (uc[4] < 0x80 || uc[4] > 0xbf) { return -1; }
    if (uc[5] < 0x80 || uc[5] > 0xbf) { return -1; }
    *index = ((uc[0] & 0x01) << 30) | ((uc[1] & 0x3f) << 24) | ((uc[2] & 0x3f) << 18) | ((uc[3] & 0x3f) << 12) | ((uc[4] & 0x3f) << 6) | (uc[5] & 0x3f);
    return anzz;
  }

  return -1;
} /* Ende next_utf8 */


/* SML3_latin_isutf8 [reentrant]:
 * gibt zurueck, ob String UTF-8 kodiert ist
 * 1.Arg: String
 * 2.Arg: Stringlaenge
 * Rueckgabe: 1 = UTF-8 kodiert
 *            0 = nicht UTF-8 kodiert
 */
int
SML3_latin_isutf8(const char *string, size_t ssize)
{
  int anzz, index;

  if (string == NULL || ssize == 0) { return 0; }

  while ((anzz = next_utf8(string, ssize, &index)) > 0) {
    string += anzz;
    ssize -= anzz;
  }
  if (anzz < 0) { return 0; }
  return 1;
} /* Ende SML3_latin_isutf8 */


/* SML3_latin_index [thread-sicher]:
 * gibt Zeichen gemaess Indexzahl zurueck
 * 1.Arg: IO-Flag
 *        - eins aus SML3_LATIN_OUTPUT_*
 *          (falls SML3_LATIN_OUTPUT_{UP|LW}CASE und 2.Arg im ISO-Zeichensatz,
 *           muss SML3_LATIN_INPUT_ISO angegeben werden
 *           fuer Rueckgabe im ISO-Zeichensatz)
 *        - evtl. aus SML3_LATIN_MODIF_*
 * 2.Arg: Indexzahl
 * 3.Arg: fuer Rueckgabe Zeichenstring (ohne Ende-0)
 * 4.Arg: Uebergabe=sizeof(3.Arg), Rueckgabe=Anzahl Bytes im 3.Arg
 * Rueckgabe: 1 = OK oder 0 = Indexzahl ungueltig
 */
int
SML3_latin_index(int ioflag, int index, char *rzch, size_t *rsize)
{
  int retw, ranz;
  const char *rtab;
  struct tabrets tbr;

  if (rzch == NULL || rsize == NULL || *rsize == 0) { return 0; }

  retw = 0;
  rtab = idx_tab(index, ioflag, &ranz, &tbr);

  if (rtab != NULL) {
    if (ranz < (int)*rsize) { *rsize = ranz; }
    memmove(rzch, rtab, *rsize);
    retw = 1;
  } else {
    *rsize = 0; *rzch = '\0'; retw = 0;
  }

  return retw;
} /* Ende SML3_latin_index */


/* SML3_latin_next [thread-sicher]:
 * gibt naechstes Zeichen zurueck
 * 1.Arg: IO-Flag
 *        - eins aus SML3_LATIN_INPUT_*
 *          (wenn fehlt, wird UTF-8 verwendet, und eventuelle ISO-Zeichen akzeptiert)
 *        - eins aus SML3_LATIN_OUTPUT_*
 *        - evtl. aus SML3_LATIN_MODIF_*
 * 2.Arg: String
 * 3.Arg: Stringlaenge
 * 4.Arg: fuer Rueckgabe Zeichenstring (ohne Ende-0)
 * 5.Arg: Uebergabe=sizeof(4.Arg), Rueckgabe=Anzahl Bytes im 4.Arg
 * Rueckgabe: Anzahl verwendeter Bytes im String
 *            oder 0  = Ende
 *            oder -1 = ungueltiges Byte bzw. Zeichensequenz nicht beendet
 */
int
SML3_latin_next(int ioflag, const char *string, size_t ssize, char *rzch, size_t *rsize)
{
  int anzz, index, ranz;
  const char *rtab;
  struct tabrets tbr;

  if (string == NULL || ssize == 0) { return 0; }

  if (ioflag & SML3_LATIN_INPUT_UTF8) {
    anzz = next_utf8(string, ssize, &index);
    if (anzz <= 0) { return anzz; }
  } else if (ioflag & SML3_LATIN_INPUT_ISO) {
    anzz = 1;
    index = (int)(unsigned char)string[0];
  } else {
    anzz = next_utf8(string, ssize, &index);
    if (anzz == 0) { return anzz; }
    if (anzz < 0) { anzz = 1; index = (int)(unsigned char)string[0]; }
  }

  rtab = idx_tab(index, ioflag, &ranz, &tbr);
  if (rzch != NULL && rsize != NULL && *rsize > 0) {
    if (rtab != NULL) {
      if (ranz < (int)*rsize) { *rsize = ranz; }
      memmove(rzch, rtab, *rsize);
    } else {
      if (ioflag & SML3_LATIN_OUTPUT_ASCII) {
        *rsize = 1;
        *rzch = '?';
      } else if (ioflag & SML3_LATIN_OUTPUT_ISO) {
        *rsize = 1;
        *rzch = '?';
      } else if (ioflag & SML3_LATIN_OUTPUT_SORT) {
        if (anzz < (int)*rsize) { *rsize = anzz; }
        memmove(rzch, string, *rsize);
      } else if (ioflag & (SML3_LATIN_OUTPUT_UPCASE | SML3_LATIN_OUTPUT_LWCASE)) {
        if (anzz < (int)*rsize) { *rsize = anzz; }
        memmove(rzch, string, *rsize);
      } else {
        *rsize = 1;
        *rzch = '?';
      }
    }
  }

  return anzz;
} /* Ende SML3_latin_next */


/* SML3_latin_convert [thread-sicher]:
 * konvertiert String
 * 1.Arg: IO-Flag
 *        - eins aus SML3_LATIN_INPUT_*
 *          (wenn fehlt, wird UTF-8 verwendet, und eventuelle ISO-Zeichen akzeptiert)
 *        - eins aus SML3_LATIN_OUTPUT_*
 *        - evtl. aus SML3_LATIN_MODIF_*
 * 2.Arg: String
 * 3.Arg: Uebergabe: Stringlaenge
 *        Rueckgabe: Anzahl restl. nicht konvertierte Bytes
 *                   oder 0 = OK
 * 4.Arg: Rueckgabe-Gummistring fuer konvertierten String (erhaelt Ende-0)
 * 5.Arg: Beginnposition Rueckgabe-Gummistring
 * Rueckgabe: Anzahl Zeichen im Gummistring ab 5.Arg
 */
size_t
SML3_latin_convert(int ioflag, const char *string, size_t *stringlen, struct SML3_gummi *rgm, size_t gmstart)
{
  const size_t zblock = 1024;
  size_t rsize, zpos, zmax;
  int eanz;
  char rzch[32], *gptr;

  if (rgm == NULL) { SML3_fehlerexit("%s", SML3_strerror(EINVAL)); }
  if (string == NULL || stringlen == NULL || *stringlen == 0) {
    SML3_gumcpy(rgm, gmstart, "");
    return 0;
  }

  zpos = gmstart;
  zmax = zpos + 1;
  gptr = SML3_gumalloc(rgm, zmax);
  rsize = sizeof(rzch);
  while ((eanz = SML3_latin_next(ioflag, string, *stringlen, rzch, &rsize)) > 0) {
    if (zpos + rsize >= zmax) {
      while (zpos + rsize >= zmax) { zmax += zblock; }
      gptr = SML3_gumalloc(rgm, zmax);
    }
    memmove(gptr + zpos, rzch, rsize);
    zpos += rsize;
    string += eanz;
    *stringlen -= eanz;
    rsize = sizeof(rzch);
  }

  gptr[zpos] = '\0';
  return zpos - gmstart;
} /* Ende SML3_latin_convert */


/* SML3_latin_toupper [thread-sicher]:
 * macht lateinische Kleinbuchstaben gross
 * 1.Arg: ob String 2.Arg UTF-8 ist: 1 = UTF-8 oder 0 = ISO
 * 2.Arg: String
 * Rueckgabe: allozierter konvertierter String
 */
char *
SML3_latin_toupper(int obutf8, const char *string)
{
  struct SML3_gummi gm = SML3_GUM_INITIALIZER;
  size_t slen;

  if (string == NULL || *string == '\0') { return SML3_gumswapstr(&gm, NULL, 0); }

  slen = strlen(string);
  SML3_latin_convert((obutf8 ? SML3_LATIN_INPUT_UTF8 : SML3_LATIN_INPUT_ISO) | SML3_LATIN_OUTPUT_UPCASE, string, &slen, &gm, 0);

  return SML3_gumswapstr(&gm, NULL, 0);
} /* Ende SML3_latin_toupper */


/* SML3_latin_tolower [thread-sicher]:
 * macht lateinische Grossbuchstaben klein
 * 1.Arg: ob String 2.Arg UTF-8 ist: 1 = UTF-8 oder 0 = ISO
 * 2.Arg: String
 * Rueckgabe: allozierter konvertierter String
 */
char *
SML3_latin_tolower(int obutf8, const char *string)
{
  struct SML3_gummi gm = SML3_GUM_INITIALIZER;
  size_t slen;

  if (string == NULL || *string == '\0') { return SML3_gumswapstr(&gm, NULL, 0); }

  slen = strlen(string);
  SML3_latin_convert((obutf8 ? SML3_LATIN_INPUT_UTF8 : SML3_LATIN_INPUT_ISO) | SML3_LATIN_OUTPUT_LWCASE, string, &slen, &gm, 0);

  return SML3_gumswapstr(&gm, NULL, 0);
} /* Ende SML3_latin_tolower */


/* SML3_latin_strlen [thread-sicher]:
 * gibt Anzahl Zeichen eines UTF-8-Strings zurueck
 * 1.Arg: UTF-8-String
 * Rueckgabe: >=0 = Anzahl Zeichen (nicht Bytes!)
 *             -1 = 1.Arg ist kein UTF-8
 */
int
SML3_latin_strlen(const char *string)
{
  int eanz, anzz;
  size_t slen;

  if (string == NULL || *string == '\0') { return 0; }

  slen = strlen(string);
  anzz = 0;
  while ((eanz = SML3_latin_next(SML3_LATIN_INPUT_UTF8, string, slen, NULL, NULL)) > 0) {
    anzz++;
    string += eanz;
    slen -= eanz;
  }
  if (eanz < 0) { return -1; }

  return anzz;
} /* Ende SML3_latin_strlen */


/* SML3_latin_strcmp [thread-sicher]:
 * aehnlich strcmp() fuer UTF-8-Strings
 * 1.Arg: UTF-8-String
 * 2.Arg: UTF-8-String
 * 3.Arg: 0 = case-sensitive   (aehnlich strcmp())
 *        1 = case-insensitive (aehnlich strcasecmp())
 * Rueckgabe: -1 = 1.Arg ist kleiner
 *             0 = beide Arg sind gleich
 *             1 = 1.Arg ist groesser
 */
int
SML3_latin_strcmp(const char *s1, const char *s2, int icase)
{
  int e1, e2, ioflag, retw;
  size_t b1, b2, r1, r2;
  char rz1[32], rz2[32];

  if (s1 == NULL) { s1 = ""; }
  b1 = strlen(s1);
  if (s2 == NULL) { s2 = ""; }
  b2 = strlen(s2);

  ioflag = SML3_LATIN_INPUT_UTF8;
  if (icase) {
    ioflag |= SML3_LATIN_OUTPUT_LWCASE;
  } else {
    ioflag |= SML3_LATIN_OUTPUT_UTF8;
  }

  retw = 0;
  for (;;) {
    r1 = sizeof(rz1) - 1;
    r2 = sizeof(rz2) - 1;
    e1 = SML3_latin_next(ioflag, s1, b1, rz1, &r1);
    e2 = SML3_latin_next(ioflag, s2, b2, rz2, &r2);
    if (e1 <= 0 && e2 <= 0) { break; }
    if (e1 <= 0) { retw = -1; break; }
    if (e2 <= 0) { retw = 1; break; }
    s1 += e1;
    b1 -= e1;
    s2 += e2;
    b2 -= e2;
    rz1[r1] = '\0';
    rz2[r2] = '\0';
    retw = strcmp(rz1, rz2);
    if (retw != 0) {
      if (retw < 0) { retw = -1; }
      if (retw > 0) { retw = 1; }
      break;
    }
  }

  return retw;
} /* Ende SML3_latin_strcmp */
