#!/bin/sh
# config.sh: check system parameters

cd `dirname $0` || exit 1

# global variables

# PREFIX: installation prefix
test "$PREFIX" = "" && PREFIX="/usr/local"
# CC: compiler
test "$CC" = "" && CC="cc"
# LIBNAME: library name (without lib), e.g. monkey
if test "$LIBNAME" = ""
then
  LIBNAME="`basename \`pwd\` | sed -e 's/-.*//'`"
  test "$LIBNAME" = "" && { echo "Error: LIBNAME empty"; exit 1; }
fi
echo "$LIBNAME" | grep / >/dev/null && { echo "Error: LIBNAME=\"$LIBNAME\""; exit 1; }
CONF_SCRIPT="$LIBNAME-config"

# internal variables

CONFIG_H="config.h"
CONFIG_LOG="config.log"

CONFVAR_DEFINE=""
CONFVAR_LIB=""

_CONFIG_PG="_config_pg"
_PRINTF="printf"
_NWLIBS=""


if test "$1" = "-h" || test "$1" = "--help"
then
  echo "Usage: $0 [<Options>]"
  echo "Following environment variables may be defined:"
  echo " - CC: compiler (default: cc)"
  echo " - PREFIX: installation prefix (default: /usr/local)"
  echo " - LIBNAME: library name (without lib), e.g. monkey (default: {dirname})"
  echo "Writes Makefile from Makefile.template, replaces:"
  echo " - \`##PREFIX##' with \`PREFIX = \$PREFIX'"
  echo " - \`##CC##' with \`CC = \$CC'"
  echo " - \`##CF_DEFINE##' with \`CF_DEFINE = <-D Definitions>'"
  echo " - \`##CF_LIB##' with \`CF_LIB = <extra libs>'"
  echo " - \`##LIBNAME##' with \`LIBNAME = \$LIBNAME'"
  echo " - \`##CONF_SCRIPT##' with \`CONF_SCRIPT = \$LIBNAME-config'"
  echo "Writes config-file \$LIBNAME-config"
  echo
  echo "Options:"
  echo "  --prefix=PREFIX       install files in PREFIX (default: $PREFIX)"
  echo "                        (overwrites environment variable PREFIX)"
  echo "  --disable-pthread     disable Threads (default: auto)"
  echo
  exit 1
fi

# check options
opt_disable_pthread=0
while test $# -gt 0
do
  p1=`echo "$1" | cut -d= -f1`
  p2=`echo "$1" | cut -d= -s -f2`
  test "$p1" = "--prefix" && test "$p2" != "" && PREFIX="$p2"
  test "$p1" = "--disable-pthread" && opt_disable_pthread=1
  shift
done


# compiles a check program
pg_compile() {
# parameter: check infotext
  # info output
  outtext="$1"
  test "$outtext" = "" && outtext="next check"
  $_PRINTF "%-40.40s: " "$outtext"
  (echo; echo "*** $outtext ***") >> $CONFIG_LOG

  # test programm output
  rm -f $_CONFIG_PG.o $_CONFIG_PG
  (echo; echo "--> Begin test file $_CONFIG_PG.c:") >> $CONFIG_LOG
  cat $_CONFIG_PG.c >> $CONFIG_LOG
  echo "<-- End test file" >> $CONFIG_LOG

  # compiling
  (echo; echo "--> Begin compiling error messages: $CC -c $_CONFIG_PG.c") >> $CONFIG_LOG
  $CC -c $_CONFIG_PG.c 2>> $CONFIG_LOG
  cerg=$?
  echo "<-- End compiling error messages" >> $CONFIG_LOG
  if test $cerg -ne 0
  then
    rm -f $_CONFIG_PG.o
  fi
  return $cerg
}


# links a compiled check program
pg_link() {
# parameter: libraries to link with or empty
  auxlibs="$1"
  (echo; echo "--> Begin linking error messages: $CC $_CONFIG_PG.o $auxlibs $_NWLIBS -o $_CONFIG_PG") >> $CONFIG_LOG
  $CC $_CONFIG_PG.o $auxlibs $_NWLIBS -o $_CONFIG_PG 2>> $CONFIG_LOG
  cerg=$?
  echo "<-- End linking error messages" >> $CONFIG_LOG
  if test $cerg -ne 0
  then
    rm -f $_CONFIG_PG.o $_CONFIG_PG
    return $cerg
  fi
  return $cerg
}


# runs a linked check program
pg_exec() {
# return variable pgout contains output of check-program
  (echo; echo "--> Begin executing ./$_CONFIG_PG") >> $CONFIG_LOG
  pgout=`./$_CONFIG_PG 2>> $CONFIG_LOG`
  cerg=$?
  echo "<-- End executing with return status $cerg" >> $CONFIG_LOG
  rm -f $_CONFIG_PG.o $_CONFIG_PG
  return $cerg
}



# ***************************************
# ***** Check for system parameters *****
# ***************************************

echo
echo "==> Common config tests <=="
echo
echo "PREFIX=\"$PREFIX\""
echo "CC=\"$CC\""
echo "LIBNAME=\"$LIBNAME\""
echo
echo "==> Testing your system ..."
rm -f $CONFIG_H $CONFIG_LOG
echo "#ifndef SML3__CONFIG_H_" > $CONFIG_H
echo "#define SML3__CONFIG_H_" >> $CONFIG_H
echo >> $CONFIG_H

echo "###########################" >> $CONFIG_LOG
echo "### Common Config Tests ###" >> $CONFIG_LOG
echo "###########################" >> $CONFIG_LOG
echo >> $CONFIG_LOG
echo "PREFIX=\"$PREFIX\"" >> $CONFIG_LOG
echo "CC=\"$CC\"" >> $CONFIG_LOG
echo >> $CONFIG_LOG


# ++++++++++++++++++++++++++++++++++++++
# +++ check for additional libraries +++
# ++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
int
main(void)
{
  return 0;
}
__EOF__

nwlibs=""
# SYSV libraries not checked
#for nwlib in "-lsocket" "-lnsl"
#do
#  rm -f $_CONFIG_PG.o $_CONFIG_PG
#  pg_compile "Have additional library \"$nwlib\"" && pg_link "$nwlib"
#  if test $? -eq 0
#  then
#    echo "yes"
#    (echo; echo "==> yes <==") >> $CONFIG_LOG
#    CONFVAR_LIB="$CONFVAR_LIB $nwlib"
#    nwlibs="$nwlibs $nwlib"
#  else
#    echo "no"
#    (echo; echo "==> no <==") >> $CONFIG_LOG
#  fi
#done
#rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
#(echo; echo "####################") >> $CONFIG_LOG
_NWLIBS="$nwlibs"


# ++++++++++++++++++++++++++++++++++
# +++ 8-byte integer: SML3_int64 +++
# ++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
int
main(void)
{
  if (sizeof(long) == 8) { printf("1\n"); return 0; }
  if (sizeof(long long) == 8) { printf("2\n"); return 0;}
  printf("0\n");
  return 0;
}
__EOF__

pg_compile "8-byte integer (SML3_int64)" && pg_link "" && pg_exec
if test $? -ne 0 
then
  rm -f $_CONFIG_PG.c
  echo "- error (see $CONFIG_LOG)"
  echo "==> Error: aborting." >> $CONFIG_LOG
  rm -f $CONFIG_H
  exit 1
fi
rm -f $_CONFIG_PG.c

echo >> $CONFIG_H
echo "/* Installations-Prefix, Programme liegen unter SML3_PREFIX/bin */" >> $CONFIG_H
echo "#define SML3_PREFIX \"$PREFIX\"" >> $CONFIG_H
echo >> $CONFIG_H

if test "$pgout" = "1"
then
  echo "long"
  (echo; echo "==> long <==") >> $CONFIG_LOG
  echo "/* SML3_int64: 8-byte integer */" >> $CONFIG_H
  echo "#define SML3_int64 long" >> $CONFIG_H
  echo "/* SMLINT64_FORMAT: printf()-Format for SML3_int64 */" >> $CONFIG_H
  echo "#define SMLINT64_FORMAT \"l\"" >> $CONFIG_H
  echo "/* SMLINT64_MAX: max. Value for signed SML3_int64 */" >> $CONFIG_H
  echo "#define SMLINT64_MAX 0x7fffffffffffffffL" >> $CONFIG_H
  echo "/* SMLINT64_MIN: min. Value for signed SML3_int64 */" >> $CONFIG_H
  echo "#define SMLINT64_MIN (-0x7fffffffffffffffL - 1L)" >> $CONFIG_H
  echo "/* SMLINT64_UMAX: max. Value for unsigned SML3_int64 */" >> $CONFIG_H
  echo "#define SMLINT64_UMAX 0xffffffffffffffffUL" >> $CONFIG_H
elif test "$pgout" = "2"
then
  echo "long long"
  (echo; echo "==> long long <==") >> $CONFIG_LOG
  echo "/* SML3_int64: 8-byte integer */" >> $CONFIG_H
  echo "#define SML3_int64 long long" >> $CONFIG_H
  echo "/* SMLINT64_FORMAT: printf()-Format for SML3_int64 */" >> $CONFIG_H
  echo "#define SMLINT64_FORMAT \"ll\"" >> $CONFIG_H
  echo "/* SMLINT64_MAX: max. Value for signed SML3_int64 */" >> $CONFIG_H
  echo "#define SMLINT64_MAX 0x7fffffffffffffffLL" >> $CONFIG_H
  echo "/* SMLINT64_MIN: min. Value for signed SML3_int64 */" >> $CONFIG_H
  echo "#define SMLINT64_MIN (-0x7fffffffffffffffLL - 1LL)" >> $CONFIG_H
  echo "/* SMLINT64_UMAX: max. Value for unsigned SML3_int64 */" >> $CONFIG_H
  echo "#define SMLINT64_UMAX 0xffffffffffffffffULL" >> $CONFIG_H
else
  echo "not available, bad"
  echo "==> not available: aborting." >> $CONFIG_LOG
  rm -f $CONFIG_H
  exit 1
fi
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++
# +++ find endian +++
# +++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
int
main(void)
{
  union {
    short a;
    char b[2];
  } ue;
  ue.a = 1;
  if (ue.b[0] == 1) { printf("L\n"); } else { printf("B\n"); }
  return 0;
}
__EOF__

pg_compile "Endian" && pg_link "" && pg_exec
if test $? -ne 0 
then
  rm -f $_CONFIG_PG.c
  echo "- error (see $CONFIG_LOG)"
  echo "==> Error: aborting." >> $CONFIG_LOG
  rm -f $CONFIG_H
  exit 1
fi
rm -f $_CONFIG_PG.c

echo >> $CONFIG_H
echo "/* SML3_ENDIAN_IS_LITTLE or SML3_ENDIAN_IS_BIG */" >> $CONFIG_H
if test "$pgout" = "L"
then
  echo "Little-Endian"
  (echo; echo "==> Little-Endian <==") >> $CONFIG_LOG
  echo "#define SML3_ENDIAN_IS_LITTLE" >> $CONFIG_H
else
  echo "Big-Endian"
  (echo; echo "==> Big-Endian <==") >> $CONFIG_LOG
  echo "#define SML3_ENDIAN_IS_BIG" >> $CONFIG_H
fi
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++++++++++
# +++ check for pipe buffer size +++
# ++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
static volatile int sig_alrm = 0;
void
do_alrm(int signum)
{
  sig_alrm = 1;
}
int
main(void)
{
  char buf[1024];
  size_t blen;
  ssize_t anz;
  int fd[2];
  struct sigaction sa;
  memset(buf, '0', sizeof(buf));
  memset(&sa, 0, sizeof(sa));
  sa.sa_handler = do_alrm;
  sigaction(SIGALRM, &sa, NULL);
  if (pipe(fd) < 0) { return 1; }
  blen = 0;
  sig_alrm = 0;
  alarm(3);
  for (;;) {
    anz = write(fd[1], buf, sizeof(buf));
    if (sig_alrm) { break; }
    blen += anz;
  }
  printf("%d\n", (int)blen);
  return 0;
}
__EOF__

pg_compile "Pipe buffer size" && pg_link "" && pg_exec
if test $? -ne 0 
then
  rm -f $_CONFIG_PG.c
  echo "- error (see $CONFIG_LOG)"
  echo "==> Error: aborting." >> $CONFIG_LOG
  rm -f $CONFIG_H
  exit 1
fi
rm -f $_CONFIG_PG.c

echo >> $CONFIG_H
echo "/* Pipe buffer size in bytes */" >> $CONFIG_H
echo "$pgout bytes"
(echo; echo "==> $pgout bytes <==") >> $CONFIG_LOG
echo "#define SML3_PIPE_BUFFERSIZE $pgout" >> $CONFIG_H
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++++++
# +++ check for sys/select.h +++
# ++++++++++++++++++++++++++++++

var_sys_select_h=""

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <sys/types.h>
#include <sys/select.h>
int
main(void)
{
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have sys/select.h */" >> $CONFIG_H

pg_compile "Have sys/select.h" && pg_link "" && pg_exec
if test $? -ne 0 
then
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_SYS_SELECT_H */" >> $CONFIG_H
  var_sys_select_h=""
else
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_SYS_SELECT_H" >> $CONFIG_H
  var_sys_select_h="#include <sys/select.h>"
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++++++++
# +++ get time select(2) needs +++
# ++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
$var_sys_select_h
int
main(void)
{
  struct timeval tv;
  struct timezone tz;
  int ms, ws, i1;
  if (gettimeofday(&tv, &tz) !=0 ) { return 1; }
  ms = (tv.tv_sec % 60) * 1000 + tv.tv_usec / 1000;
  for (i1 = 0; i1 < 100; i1++) {
    tv.tv_usec = 10000;
    tv.tv_sec = 0;
    select(0, NULL, NULL, NULL, &tv);
  }
  if (gettimeofday(&tv, &tz) != 0) { return 1; }
  ws = (tv.tv_sec % 60) * 1000 + tv.tv_usec / 1000;
  if (ws < ms) { ws += 60000; }
  ws -= ms;
  ws -= 1000;
  ws /= 100;
  printf("%d\n", ws);
  return 0;
}
__EOF__

pg_compile "Time in msec select(2) needs" && pg_link "" && pg_exec
if test $? -ne 0 
then
  rm -f $_CONFIG_PG.c
  echo "- error (see $CONFIG_LOG)"
  echo "==> Error: aborting." >> $CONFIG_LOG
  rm -f $CONFIG_H
  exit 1
fi
rm -f $_CONFIG_PG.c

echo >> $CONFIG_H
echo "/* time in msec select(2) needs */" >> $CONFIG_H
echo "$pgout milliseconds"
(echo; echo "==> $pgout milliseconds <==") >> $CONFIG_LOG
echo "#define SML3_TIME_OF_SELECT $pgout" >> $CONFIG_H
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++
# +++ have dlopen? +++
# ++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <dlfcn.h>
int
main(void)
{
  void *hdl;
  hdl = dlopen("test", RTLD_NOW);
  dlerror();
  hdl = dlsym(hdl, "funk");
  dlclose(hdl);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have dlopen */" >> $CONFIG_H

pg_compile "Have dlopen" && pg_link ""
if test $? -ne 0 
then
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  rm -f $_CONFIG_PG.o $_CONFIG_PG
  pg_compile "Have dlopen with -ldl" && pg_link "-ldl"
  if test $? -ne 0 
  then
    echo "no"
    (echo; echo "==> no <==") >> $CONFIG_LOG
    echo "/* #undef SML3_HAVE_DLOPEN */" >> $CONFIG_H
  else
    echo "yes"
    (echo; echo "==> yes <==") >> $CONFIG_LOG
    echo "#define SML3_HAVE_DLOPEN" >> $CONFIG_H
    CONFVAR_LIB="$CONFVAR_LIB -ldl"
  fi
else
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_DLOPEN" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++
# +++ have STREAMS? +++
# +++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <stropts.h>
int
main(void)
{
  ioctl(0, I_LIST, (void *)0);
  if (isastream(-2) >=0 ) { printf("0\n"); }
  printf("1\n");
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have STREAMS */" >> $CONFIG_H

pg_compile "Have STREAMS" && pg_link "" && pg_exec
if test $? -eq 0 && test "$pgout" = "1"
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_STREAMS" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_STREAMS */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG



# +++++++++++++++++++++++++
# +++ check for va_copy +++
# +++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <stdarg.h>
int
main(void)
{
  va_list ap, aq;
  va_copy(aq, ap);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have va_copy */" >> $CONFIG_H

pg_compile "Have va_copy" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "/* #undef va_copy __va_copy */" >> $CONFIG_H
else
  echo "no - assuming __va_copy"
  (echo; echo "==> no  - assuming __va_copy<==") >> $CONFIG_LOG
  echo "#define va_copy __va_copy" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++
# +++ check for strlcpy +++
# +++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
int
main(void)
{
  char buf[32] = "";
  strlcpy(buf, "123", sizeof(buf));
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have strlcpy */" >> $CONFIG_H

pg_compile "Have strlcpy" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_STRLCPY" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_STRLCPY */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++
# +++ check for strlcat +++
# +++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
int
main(void)
{
  char buf[32] = "";
  strlcat(buf, "123", sizeof(buf));
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have strlcat */" >> $CONFIG_H

pg_compile "Have strlcat" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_STRLCAT" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_STRLCAT */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++
# +++ check for fnmatch +++
# +++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <fnmatch.h>
int
main(void)
{
  fnmatch("*", "file", FNM_NOESCAPE | FNM_PATHNAME | FNM_PERIOD);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have fnmatch */" >> $CONFIG_H

pg_compile "Have fnmatch" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_FNMATCH" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_FNMATCH */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++
# +++ check for regex +++
# +++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <regex.h>
int
main(void)
{
  regex_t rx;
  if (regcomp(&rx, "Test.[0-9]?ing", REG_EXTENDED | REG_ICASE | REG_NEWLINE) != 0) { return 1; }
  regfree(&rx);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have regex */" >> $CONFIG_H

pg_compile "Have regex" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
else
  echo "not available, bad"
  echo "==> not available: aborting." >> $CONFIG_LOG
  rm -f $CONFIG_H $_CONFIG_PG.c
  exit 1
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++
# +++ check for pselect +++
# +++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <sys/select.h>
// also check for prototype conflict
extern int pselect(int, fd_set *, fd_set *, fd_set *, const struct timespec *, const sigset_t *);
int
main(void)
{
  struct timespec ts;
  ts.tv_sec = 0; ts.tv_nsec = 1;
  pselect(0, NULL, NULL, NULL, &ts, NULL);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have pselect */" >> $CONFIG_H

pg_compile "Have pselect" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_PSELECT" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_PSELECT */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++
# +++ check for ppoll +++
# +++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <poll.h>
int
main(void)
{
  extern int ppoll(struct pollfd *, nfds_t, const struct timespec *, const sigset_t *);
  struct timespec ts;
  struct pollfd fds = {0};
  ts.tv_sec = 0; ts.tv_nsec = 1;
  fds.fd = -1;
  ppoll(&fds, 1, &ts, NULL);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have ppoll */" >> $CONFIG_H

pg_compile "Have ppoll" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_PPOLL" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_PPOLL */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++
# +++ have epoll or kqueue? +++
# +++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <sys/epoll.h>
int
main(void) {
  int evfd = epoll_create(1);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have epoll */" >> $CONFIG_H

pg_compile "Have epoll" && pg_link ""
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_EPOLL" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_EPOLL */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
int
main(void) {
  int evfd = kqueue();
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have kqueue */" >> $CONFIG_H

pg_compile "Have kqueue" && pg_link ""
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_KQUEUE" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_KQUEUE */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++
# +++ check for daemon +++
# ++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
int
main(void)
{
  int i = daemon(1, 1);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have daemon */" >> $CONFIG_H

pg_compile "Have daemon" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_DAEMON" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_DAEMON */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++
# +++ check for /dev/ptmx +++
# +++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
int
main(void)
{
  if (open("/dev/ptmx", O_RDWR) < 0) { return 1; }
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have /dev/ptmx */" >> $CONFIG_H

pg_compile "Have /dev/ptmx" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_DEV_PTMX" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_DEV_PTMX */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++
# +++ check for ptsname +++
# +++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
int
main(void)
{
  char *ptr = ptsname(0);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have ptsname */" >> $CONFIG_H

pg_compile "Have ptsname" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_PTSNAME" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_PTSNAME */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++
# +++ check for grantpt +++
# +++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
int
main(void)
{
  int i = grantpt(0);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have grantpt */" >> $CONFIG_H

pg_compile "Have grantpt" && pg_link ""
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_GRANTPT" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_GRANTPT */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++
# +++ check for unlockpt +++
# ++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
int
main(void)
{
  int i = unlockpt(0);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have unlockpt */" >> $CONFIG_H

pg_compile "Have unlockpt" && pg_link ""
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_UNLOCKPT" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_UNLOCKPT */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++++++
# +++ check for sys/sockio.h +++
# ++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/sockio.h>
int
main(void)
{
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have sys/sockio.h */" >> $CONFIG_H

pg_compile "Have sys/sockio.h" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_SYS_SOCKIO_H" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_SYS_SOCKIO_H */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++++++++++++++++
# +++ check for sa_len in struct sockaddr +++
# +++++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
int
main(void)
{
  size_t i = sizeof(((struct sockaddr *)0)->sa_len);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have sa_len in struct sockaddr */" >> $CONFIG_H

pg_compile "Have sa_len in struct sockaddr" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_SOCKADDR_SA_LEN" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_SOCKADDR_SA_LEN */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++++++++++++++
# +++ check for ifr_mtu in struct ifreq +++
# +++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
int
main(void)
{
  size_t i = sizeof(((struct ifreq *)0)->ifr_mtu);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have ifr_mtu in struct ifreq */" >> $CONFIG_H

pg_compile "Have ifr_mtu in struct ifreq" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_STRUCT_IFREQ_IFR_MTU" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_STRUCT_IFREQ_IFR_MTU */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++++++++++++++++
# +++ check for ifr_index in struct ifreq +++
# +++++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
int
main(void)
{
  size_t i = sizeof(((struct ifreq *)0)->ifr_index);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have ifr_index in struct ifreq */" >> $CONFIG_H

pg_compile "Have ifr_index in struct ifreq" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_STRUCT_IFREQ_IFR_INDEX" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_STRUCT_IFREQ_IFR_INDEX */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++++++++++++++++++
# +++ check for ifr_ifindex in struct ifreq +++
# +++++++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
int
main(void)
{
  size_t i = sizeof(((struct ifreq *)0)->ifr_ifindex);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have ifr_ifindex in struct ifreq */" >> $CONFIG_H

pg_compile "Have ifr_ifindex in struct ifreq" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_STRUCT_IFREQ_IFR_IFINDEX" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_STRUCT_IFREQ_IFR_IFINDEX */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++++++++++++++++++++
# +++ check for ifr_hwaddr in struct ifreq +++
# ++++++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
int
main(void)
{
  size_t i = sizeof(((struct ifreq *)0)->ifr_hwaddr);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have ifr_hwaddr in struct ifreq */" >> $CONFIG_H

pg_compile "Have ifr_hwaddr in struct ifreq" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_STRUCT_IFREQ_IFR_HWADDR" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_STRUCT_IFREQ_IFR_HWADDR */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++++++++++++++++++++
# +++ check for ifr_enaddr in struct ifreq +++
# ++++++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
int
main(void)
{
  size_t i = sizeof(((struct ifreq *)0)->ifr_enaddr);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have ifr_enaddr in struct ifreq */" >> $CONFIG_H

pg_compile "Have ifr_enaddr in struct ifreq" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_STRUCT_IFREQ_IFR_ENADDR" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_STRUCT_IFREQ_IFR_ENADDR */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++++++++++++++++++++
# +++ check for struct ifaddrs and getifaddrs +++
# +++++++++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
#include <ifaddrs.h>
int
main(void)
{
  size_t i = sizeof(((struct ifaddrs *)0)->ifa_name);
  struct ifaddrs *ifas;
  getifaddrs(&ifas);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have struct ifaddrs and getifaddrs */" >> $CONFIG_H

pg_compile "Have struct ifaddrs and getifaddrs" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_STRUCT_IFADDRS" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_STRUCT_IFADDRS */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++++++++++++++++
# +++ check for if_nameindex functions +++
# ++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
int
main(void)
{
  unsigned int i = if_nametoindex("");
  struct if_nameindex *iptr = if_nameindex();
  if (iptr != NULL) { if_freenameindex(iptr); }
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have if_nameindex functions */" >> $CONFIG_H

pg_compile "Have if_nameindex functions" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_NAMEINDEX" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_NAMEINDEX */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++
# +++ check for net/if_dl.h +++
# +++++++++++++++++++++++++++++

var_net_if_dl_h=""

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
#include <net/if_dl.h>
int
main(void)
{
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have net/if_dl.h */" >> $CONFIG_H

pg_compile "Have net/if_dl.h" && pg_link "" && pg_exec
if test $? -ne 0 
then
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_NET_IF_DL_H */" >> $CONFIG_H
  var_net_if_dl_h=""
else
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_NET_IF_DL_H" >> $CONFIG_H
  var_net_if_dl_h="#include <net/if_dl.h>"
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++++++++++++
# +++ check for struct sockaddr_dl +++
# ++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
$var_net_if_dl_h
int
main(void)
{
  size_t i = sizeof(((struct sockaddr_dl *)0)->sdl_index);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have struct sockaddr_dl */" >> $CONFIG_H

pg_compile "Have struct sockaddr_dl" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_SOCKADDR_DL_STRUCT" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_SOCKADDR_DL_STRUCT */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++
# +++ check for getaddrinfo +++
# +++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
int
main(void)
{
  getaddrinfo(NULL, NULL, NULL, NULL);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have getaddrinfo */" >> $CONFIG_H

pg_compile "Have getaddrinfo" && pg_link ""
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_GETADDRINFO" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_GETADDRINFO */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++
# +++ check for inet_pton +++
# +++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
int
main(void)
{
  inet_pton(0, NULL, NULL);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have inet_pton */" >> $CONFIG_H

pg_compile "Have inet_pton" && pg_link ""
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_INET_PTON" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_INET_PTON */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++
# +++ check for inet_ntop +++
# +++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
int
main(void)
{
  inet_ntop(0, NULL, NULL, 0);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have inet_ntop */" >> $CONFIG_H

pg_compile "Have inet_ntop" && pg_link ""
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_INET_NTOP" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_INET_NTOP */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++
# +++ check for msg_control +++
# +++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
int
main(void)
{
  struct msghdr h;
  h.msg_control = 0;
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have msg_control */" >> $CONFIG_H

pg_compile "Have msg_control" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_MSG_CONTROL" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_MSG_CONTROL */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++
# +++ check for socklen_t +++
# +++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
int
main(void)
{
  socklen_t s;
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have socklen_t */" >> $CONFIG_H

pg_compile "Have socklen_t" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "#define socklen_t int" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++++++++++++++
# +++ check for struct sockaddr_storage +++
# +++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
int
main(void)
{
  struct sockaddr_storage s;
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have struct sockaddr_storage */" >> $CONFIG_H

pg_compile "Have struct sockaddr_storage" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_SOCKADDR_STORAGE" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_SOCKADDR_STORAGE */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++++++++++++++++++++++++++++
# +++ check for getpw{nam|uid}_r, getgr{nam|gid}_r +++
# ++++++++++++++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
int
main(void)
{
  struct passwd pw, *pwp;
  struct group gr, *grp;
  char buf[1024];
  getpwnam_r("root", &pw, buf, sizeof(buf), &pwp);
  getpwuid_r(0, &pw, buf, sizeof(buf), &pwp);
  getgrnam_r("wheel", &gr, buf, sizeof(buf), &grp);
  getgrgid_r(0, &gr, buf, sizeof(buf), &grp);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have getpwnam_r, getpwuid_r, getgrnam_r, getgrgid_r */" >> $CONFIG_H

pg_compile "Have get{pw|gr}???_r" && pg_link "" && pg_exec
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_GETPWGR_R" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_GETPWGR_R */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++
# +++ check for pthread +++
# +++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <pthread.h>
int
main(void)
{
  pthread_t th1 = pthread_self();
  pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutexattr_t rma;
  pthread_mutex_lock(&mutex);
  pthread_mutex_unlock(&mutex);
  pthread_mutexattr_init(&rma);
  pthread_mutexattr_settype(&rma, PTHREAD_MUTEX_RECURSIVE);
  pthread_mutexattr_destroy(&rma);
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have pthread */" >> $CONFIG_H

pg_compile "Have pthread" && pg_link "-lpthread" && pg_exec
if test $? -eq 0
then
  if test $opt_disable_pthread -gt 0
  then
    echo "yes - disabled via option"
    (echo; echo "==> yes - disabled via option <==") >> $CONFIG_LOG
    echo "/* #undef SML3_HAVE_PTHREAD */" >> $CONFIG_H
  else
    echo "yes"
    (echo; echo "==> yes <==") >> $CONFIG_LOG
    CONFVAR_LIB="$CONFVAR_LIB -lpthread"
    echo "#define SML3_HAVE_PTHREAD" >> $CONFIG_H
  fi
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "/* #undef SML3_HAVE_PTHREAD */" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG


# +++++++++++++++++++++++++++++++++++++++++
# +++ check for shm_open and shm_unlink +++
# +++++++++++++++++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
int
main(void)
{
  int fd;
  char buf[128];
  snprintf(buf, sizeof(buf), "/tmp/sml2.%d", (int)getpid());
  fd = shm_open(buf, O_RDWR | O_CREAT, 0666);
  if (fd >= 0) { close(fd); shm_unlink(buf); }
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have shm_open and shm_unlink */" >> $CONFIG_H

pg_compile "Have shm_open and shm_unlink" && pg_link ""
if test $? -ne 0 
then
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  rm -f $_CONFIG_PG.o $_CONFIG_PG
  pg_compile "Have shm_open and shm_unlink with -lrt" && pg_link "-lrt"
  if test $? -ne 0 
  then
    echo "no"
    (echo; echo "==> no <==") >> $CONFIG_LOG
    echo "/* #undef SML3_HAVE_SHM_OPEN */" >> $CONFIG_H
  else
    echo "yes"
    (echo; echo "==> yes <==") >> $CONFIG_LOG
    echo "#define SML3_HAVE_SHM_OPEN" >> $CONFIG_H
    CONFVAR_LIB="$CONFVAR_LIB -lrt"
  fi
else
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_HAVE_SHM_OPEN" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG


# ++++++++++++++++++++++++++
# +++ check for __func__ +++
# ++++++++++++++++++++++++++

rm -f $_CONFIG_PG.c
cat > $_CONFIG_PG.c <<__EOF__
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
int
main(void)
{
#define SML3_funcname __func__
  return 0;
}
__EOF__

echo >> $CONFIG_H
echo "/* have __func__ */" >> $CONFIG_H

pg_compile "Have __func__" && pg_link ""
if test $? -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> $CONFIG_LOG
  echo "#define SML3_funcname __func__" >> $CONFIG_H
else
  echo "no"
  (echo; echo "==> no <==") >> $CONFIG_LOG
  echo "#define SML3_funcname \"\"" >> $CONFIG_H
fi
rm -f $_CONFIG_PG.c $_CONFIG_PG.o $_CONFIG_PG
(echo; echo "####################") >> $CONFIG_LOG


# ***********************
# ***** Checks done *****
# ***********************

(echo; echo "#endif /* SML3__CONFIG_H_ */") >> $CONFIG_H

CONFVAR_LIB="$CONFVAR_LIB -lm"
CONFVAR_DEFINE="$CONFVAR_DEFINE -DPREFIX=\\\\\"$PREFIX\\\\\""

echo
echo "Creating depends for Makefile ..."
sh dep.sh Makefile.template Makefile.dep || { echo "==> Error: aborting."; rm -f $CONFIG_H; exit 1; }

echo "Creating Makefile ..."
cat Makefile.dep \
  | sed -e "s%##PREFIX##%PREFIX = $PREFIX%" \
  | sed -e "s%##CC##%CC = $CC%" \
  | sed -e "s%##CF_DEFINE##%CF_DEFINE = $CONFVAR_DEFINE%" \
  | sed -e "s%##CF_LIB##%CF_LIB = $CONFVAR_LIB%" \
  | sed -e "s%##LIBNAME##%LIBNAME = $LIBNAME%" \
  | sed -e "s%##CONF_SCRIPT##%CONF_SCRIPT = $CONF_SCRIPT%" \
  > Makefile
rm -f Makefile.dep

echo "Creating $CONF_SCRIPT ..."

cat << __EOCONF1__ > $CONF_SCRIPT
#!/bin/sh

libname="$LIBNAME"
incdir="$PREFIX/include"
libdir="$PREFIX/lib"
prefix="$PREFIX"
strixmod="$PREFIX/lib/strixmod"
extralibs="$CONFVAR_LIB"
__EOCONF1__

cat << '__EOCONF2__' >> $CONF_SCRIPT

para=""
while test $# -gt 0
do
  if test "$1" = "--cflags"
  then
    # -I
    para="$para -I$incdir"

  elif test "$1" = "--libs"
  then
    # -L and -l
    para="$para -L$libdir -l$libname $extralibs"

  elif test "$1" = "--libspath"
  then
    # -L and -l, library path hardcoded
    para="$para -L$libdir -Wl,-rpath,$libdir -l$libname $extralibs"

  elif test "$1" = "--extralibs"
  then
    # extra libraries
    echo "$extralibs"
    para=":"

  elif test "$1" = "--prefix"
  then
    # install prefix
    echo "$prefix"
    para=":"

  elif test "$1" = "--strixmod"
  then
    # standard modul path for strix-moduls
    echo "$strixmod"
    para=":"
  fi
  shift
done

if test "$para" != ""
then
  test "$para" != ":" && echo " $para"
else
  echo "Usage: $0 <--cflags|--libs|--libspath>" >&2
  echo "       $0 --extralibs" >&2
  echo "       $0 --prefix" >&2
  echo "       $0 --strixmod" >&2
  exit 1
fi
exit 0
__EOCONF2__
chmod 755 $CONF_SCRIPT

echo
echo "==> Config Tests done (see $CONFIG_LOG) <=="
echo
exit 0
