#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "vg4.h"

int cmd_sprite(int, char **);

static int cmd_sprite_show(int, char **);


/* process command "sprite" */
int
cmd_sprite(int argc, char **argv)
{
  int retw = 0;

  if (argc < 2 || strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0)  { goto help_sprite; }
  argc--; argv++;

  if (strcmp(argv[0], "show") == 0) {
    retw = cmd_sprite_show(argc, argv);
  } else {
    goto help_sprite;
  }

  return retw;

help_sprite:
  fprintf(stderr, "\nUsage: %s sprite <subcommand> [<options>]\n", argv0);
  fprintf(stderr, "Subcommands:\n");
  fprintf(stderr, "- show:  load and show sprite\n");
  fprintf(stderr, "(Use \"%s sprite <subcommand> -h|--help\" for more infos)\n", argv0);

  return 1;
} /* Ende cmd_sprite */


/* process command "sprite show" */
static int
cmd_sprite_show(int argc, char **argv)
{
  struct { int k_quit; } kref;
  int i1, wsize, gloop, audio_freq;
  VG_BOOL audio_stereo;
  const char *sprtfile;
  struct VG_Sprite *sprt;
  struct VG_Image *img; 
  struct VG_ImagecopyAttr iattr;

  if (argc < 2)  { goto help_sprite_show; }
  if (strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0)  { goto help_sprite_show; }

  opterr = opterr ? opterr : 1;

  sprtfile = NULL;
  wsize = VG_WINDOW_SIZE_LOW;
  gloop = 80;
  audio_freq = VG_AUDIO_FREQ_HIGH;
  audio_stereo = VG_TRUE;

  while ((i1 = getopt(argc, argv, "+g:f:MW")) != -1) {
    switch(i1) {
      case 'g': gloop = atoi(optarg); break;
      case 'f': if (strcmp(optarg, "low") == 0) {
                  audio_freq = VG_AUDIO_FREQ_LOW;
                } else if (strcmp(optarg, "medium") == 0) {
                  audio_freq = VG_AUDIO_FREQ_MEDIUM;
                } else if (strcmp(optarg, "high") == 0) {
                  audio_freq = VG_AUDIO_FREQ_HIGH;
                }
                break;
      case 'M': audio_stereo = VG_FALSE; break;
      case 'W': wsize = VG_WINDOW_SIZE_HIGH; break;
      default:  goto help_sprite_show;
    }
  }

  if (optind < argc) { sprtfile = argv[optind++]; }
  if (sprtfile == NULL || *sprtfile == '\0') { goto help_sprite_show; }

  /* open window */
  if (!vg4->window->open(wsize, VG_WINDOW_SCALE_BEST)) { return 1; }

  /* set keys */
  if ((kref.k_quit = vg4->input->key_insert("Quit", VG_FALSE, VG_FALSE)) == 0) { return 1; }
  vg4->input->key_setkbd(kref.k_quit, VG_INPUT_KBDCODE_Q);

  /* open audio */
  if (!vg4->audio->open(audio_freq, audio_stereo)) { return 1; }

  /* load and show sprite */
  sprt = vg4->sprite->load(sprtfile);
  if (sprt == NULL) { vg4->audio->close(); return 0; }

  /* gameloop */
  for (;;) {
    if (!vg4->input->update(VG_TRUE)) { return 1; }
    if (vg4->input->key_newpressed(kref.k_quit)) { break; }

    vg4->window->fill(VG_COLOR_RGB(64, 64, 64));

    if (!vg4->sprite->next(sprt, &img, &iattr)) { break; }

    if (img != NULL) { vg4->window->copy(img, NULL, &iattr); }

    vg4->window->flush();
    vg4->misc->wait_time(gloop);
  }

  vg4->sprite->destroy(sprt);
  vg4->input->key_remove(kref.k_quit);
  vg4->audio->close();
  vg4->window->close();

  return 0;

help_sprite_show:
  fprintf(stderr, "\nUsage: %s sprite show [<options>] <sprite-file>\n", argv0);
  fprintf(stderr, "Load and show sprite\n");
  fprintf(stderr, "Options:\n");
  fprintf(stderr, " -g <number>: gameloop in milli-seconds, (default: 80)\n");
  fprintf(stderr, " -f <string>: audio-frequency: low, medium, high, (default: high)\n");
  fprintf(stderr, " -M:          audio: mono, (default: stereo)\n");
  fprintf(stderr, " -W:          use VG_WINDOW_SIZE_HIGH, (default: VG_WINDOW_SIZE_LOW)\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "Use key \"Q\" to exit\n");
  fprintf(stderr, "\n");
  return 1;
} /* Ende cmd_sprite_show */
