#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "vg4.h"

int cmd_film(int, char **);

static int cmd_film_show(int, char **);


/* process command "film" */
int
cmd_film(int argc, char **argv)
{
  int retw = 0;

  if (argc < 2 || strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0)  { goto help_film; }
  argc--; argv++;

  if (strcmp(argv[0], "show") == 0) {
    retw = cmd_film_show(argc, argv);
  } else {
    goto help_film;
  }

  return retw;

help_film:
  fprintf(stderr, "\nUsage: %s film <subcommand> [<options>]\n", argv0);
  fprintf(stderr, "Subcommands:\n");
  fprintf(stderr, "- show:  show film\n");
  fprintf(stderr, "(Use \"%s film <subcommand> -h|--help\" for more infos)\n", argv0);

  return 1;
} /* Ende cmd_film */


/* process command "film show" */
static int
cmd_film_show(int argc, char **argv)
{
  const char *filmdir, *mlangdir;
  int i1, wsize, audio_freq;
  VG_BOOL audio_stereo, filmskip;
  struct VG_Hash *hvar;

  if (argc < 2)  { goto help_film_show; }
  if (strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0)  { goto help_film_show; }

  opterr = opterr ? opterr : 1;

  filmdir = mlangdir = NULL;
  wsize = VG_WINDOW_SIZE_LOW;
  audio_freq = VG_AUDIO_FREQ_HIGH;
  audio_stereo = VG_TRUE;
  hvar = vg4->hash->create();

  while ((i1 = getopt(argc, argv, "+m:f:V:MW")) != -1) {
    switch(i1) {
      case 'm': mlangdir = optarg; break;
      case 'f': if (strcmp(optarg, "low") == 0) {
                  audio_freq = VG_AUDIO_FREQ_LOW;
                } else if (strcmp(optarg, "medium") == 0) {
                  audio_freq = VG_AUDIO_FREQ_MEDIUM;
                } else if (strcmp(optarg, "high") == 0) {
                  audio_freq = VG_AUDIO_FREQ_HIGH;
                }
                break;
      case 'V': { const char *kptr;
                  if ((kptr = strchr(optarg, '=')) != NULL) {
                    char vkey[128], vval[256];
                    vg4->misc->strscpy(vkey, sizeof(vkey), optarg, (size_t)(kptr - optarg));
                    vg4->misc->strcpy(vval, sizeof(vval), kptr + 1);
                    if (*vkey != '\0') { vg4->hash->setstr(hvar, vkey, vval); }
                  }
                }
                break;
      case 'M': audio_stereo = VG_FALSE; break;
      case 'W': wsize = VG_WINDOW_SIZE_HIGH; break;
      default:  goto help_film_show;
    }
  }

  if (optind < argc) { filmdir = argv[optind++]; }
  if (filmdir == NULL || *filmdir == '\0') { goto help_film_show; }

  vg4->mlang->fb_locale("en");
  if (mlangdir != NULL && *mlangdir != '\0') {
    vg4->mlang->add(mlangdir);
  }

  /* open window */
  if (!vg4->window->open(wsize, VG_WINDOW_SCALE_BEST)) { return 1; }
  vg4->window->fill(VG_COLOR_RGB(64, 64, 64));

  /* open audio */
  if (!vg4->audio->open(audio_freq, audio_stereo)) { return 1; }

  /* show film */
  vg4->film->play(filmdir, NULL, &filmskip, hvar);

  vg4->hash->destroy(hvar);
  vg4->audio->close();
  vg4->window->close();

  return 0;

help_film_show:
  fprintf(stderr, "\nUsage: %s film show [<options>] <film-directory>\n", argv0);
  fprintf(stderr, "Show film\n");
  fprintf(stderr, "Options:\n");
  fprintf(stderr, " -m <string>: multilanguage directory\n");
  fprintf(stderr, " -f <string>: audio-frequency: low, medium, high, (default: high)\n");
  fprintf(stderr, " -V <string>: (repeated:) variable: <key>=<value>\n");
  fprintf(stderr, " -M:          audio: mono, (default: stereo)\n");
  fprintf(stderr, " -W:          use VG_WINDOW_SIZE_HIGH, (default: VG_WINDOW_SIZE_LOW)\n");
  fprintf(stderr, "\n");
  return 1;
} /* Ende cmd_film_show */
