#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "vg4.h"

int cmd_canvas(int, char **);

static int cmd_canvas_exec(int, char **);


/* process command "canvas" */
int
cmd_canvas(int argc, char **argv)
{
  int retw = 0;

  if (argc < 2 || strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0)  { goto help_canvas; }
  argc--; argv++;

  if (strcmp(argv[0], "exec") == 0) {
    retw = cmd_canvas_exec(argc, argv);
  } else {
    goto help_canvas;
  }

  return retw;

help_canvas:
  fprintf(stderr, "\nUsage: %s canvas <subcommand> [<options>]\n", argv0);
  fprintf(stderr, "Subcommands:\n");
  fprintf(stderr, "- exec:  load and execute canvas\n");
  fprintf(stderr, "(Use \"%s canvas <subcommand> -h|--help\" for more infos)\n", argv0);

  return 1;
} /* Ende cmd_canvas */


/* process command "canvas exec" */
static int
cmd_canvas_exec(int argc, char **argv)
{
  int i1, wsize;
  const char *cvasfile, *mlangdir, *selname;
  struct VG_Hash *hvar;
  struct VG_Canvas *cvas;

  if (argc < 2)  { goto help_canvas_exec; }
  if (strcmp(argv[1], "-h") == 0 || strcmp(argv[1], "--help") == 0)  { goto help_canvas_exec; }

  opterr = opterr ? opterr : 1;

  cvasfile = mlangdir = NULL;
  wsize = VG_WINDOW_SIZE_LOW;
  hvar = vg4->hash->create();

  while ((i1 = getopt(argc, argv, "+m:V:W")) != -1) {
    switch(i1) {
      case 'm': mlangdir = optarg; break;
      case 'V': { const char *kptr;
                  if ((kptr = strchr(optarg, '=')) != NULL) {
                    char vkey[128], vval[256];
                    vg4->misc->strscpy(vkey, sizeof(vkey), optarg, (size_t)(kptr - optarg));
                    vg4->misc->strcpy(vval, sizeof(vval), kptr + 1);
                    if (*vkey != '\0') { vg4->hash->setstr(hvar, vkey, vval); }
                  }
                }
                break;
      case 'W': wsize = VG_WINDOW_SIZE_HIGH; break;
      default:  goto help_canvas_exec;
    }
  }

  if (optind < argc) { cvasfile = argv[optind++]; }
  if (cvasfile == NULL || *cvasfile == '\0') { goto help_canvas_exec; }

  vg4->mlang->fb_locale("en");
  if (mlangdir != NULL && *mlangdir != '\0') {
    vg4->mlang->add(mlangdir);
  }

  /* open window */
  if (!vg4->window->open(wsize, VG_WINDOW_SCALE_BEST)) { return 1; }
  vg4->window->fill(VG_COLOR_RGB(64, 64, 64));

  /* load and execute canvas */
  cvas = vg4->canvas->load(cvasfile, hvar);
  vg4->hash->destroy(hvar);
  if (cvas != NULL && vg4->canvas->exec(cvas, NULL, &selname)) {
    if (selname == NULL) {
      printf("Cancelled.\n");
    } else if (*selname == '\0') {
      printf("OK, but nothing selected.\n");
    } else {
      printf("Selected item-name: %s\n", selname);
    }
  }
  if (cvas != NULL) { vg4->canvas->destroy(cvas); }

  vg4->window->close();

  return 0;

help_canvas_exec:
  fprintf(stderr, "\nUsage: %s canvas exec [<options>] <canvas-file>\n", argv0);
  fprintf(stderr, "Load and execute canvas\n");
  fprintf(stderr, "Options:\n");
  fprintf(stderr, " -m <string>: multilanguage directory\n");
  fprintf(stderr, " -V <string>: (repeated:) variable: <key>=<value>\n");
  fprintf(stderr, " -W:          use VG_WINDOW_SIZE_HIGH, (default: VG_WINDOW_SIZE_LOW)\n");
  fprintf(stderr, "\n");
  return 1;
} /* Ende cmd_canvas_exec */
