/* select player-racket */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames4.h>

VG_BOOL select_racket(struct VG_Image ***, VG_BOOL);

static void show_info(const char *);
static VG_BOOL exec_canvas_racket(char *, size_t, struct VG_NwXdataList *);


/* show info */
static void
show_info(const char *text)
{ 
  struct VG_Image *img; 

  if (text == NULL || *text == '\0') { return; }
  
  img = vg4->font->totext(text, NULL, NULL, NULL, NULL);
  vg4->window->clear();
  vg4->window->copy(img, NULL, NULL);
  vg4->window->flush();
  vg4->image->destroy(img);
} 


/* execute canvas to select player-racket
 * @param rkname     for returning name of racket
 * @param rksize     sizeof(rkname)
 * @param xdatalist  exchange-data list of already received clients
 * @return VG_TRUE = OK or VG_FALSE = got exit-request
 */
static VG_BOOL
exec_canvas_racket(char *rkname, size_t rksize, struct VG_NwXdataList *xdatalist)
{
  const char *chain_name = "rk_chain";
  struct VG_Canvas *cvas;
  const char *selname;
  struct VG_NwXdataList *xdptr;

  if (rkname == NULL || rksize == 0) { return VG_FALSE; }

  *rkname = '\0';

  /* load canvas */
  cvas = vg4->canvas->load("files/canvas/racket.cvas", NULL);
  if (cvas == NULL) { return VG_FALSE; }

  /* disable chain-elements of already set player-rackets */
  for (xdptr = xdatalist; xdptr != NULL; xdptr = xdptr->next) {
    vg4->canvas->chain_disable(cvas, chain_name, xdptr->data, VG_TRUE);
  }

  for (;;) {
    /* execute canvas */
    if (!vg4->canvas->exec(cvas, NULL, &selname)) { return VG_FALSE; }
    if (selname == NULL) { break; }  /* cancelled via escape-key or cancel-button */

    /* act according to selection */

    if (*selname == '\0') {  /* no selection, but return-key pressed */
      /* no action, re-execute canvas */
      ;

    } else if (strcmp(selname, chain_name) == 0) {  /* item selected from chain */
      const char *chainkey = vg4->canvas->chain_get_activated(cvas, selname);
      if (chainkey != NULL) {
        vg4->misc->strcpy(rkname, rksize, chainkey);
        break;
      }
    }
  }

  vg4->canvas->destroy(cvas);

  if (*rkname == '\0') { return VG_FALSE; }
  return VG_TRUE;
}


/* select a player-racket and receive selections of each clients
 * @param imgpp  for returning an array of images,
 *               the index represents the client-number (0 = client1, 1 = client2, ...),
 *               the value is the image for its client or NULL if the client has disconnected,
 *               the array's size is the number of clients
 * @param exclusive  whether player-rackets must be unique
 * @return  VG_TRUE = OK or VG_FALSE = got exit request
 */
VG_BOOL
select_racket(struct VG_Image ***imgpp, VG_BOOL exclusive)
{
  struct VG_NwXdataList *xdatalist, *xdptr;
  int idx, clnr, clmax;
  char rkname[32], filename[256];
  VG_BOOL redo;

  if (imgpp == NULL) { return VG_TRUE; }
  *imgpp = NULL;

  clnr = vg4->nw->local_clnr();
  clmax = vg4->nw->numberofclients(NULL);
  if (clnr == 0 || clmax == 0) { return VG_TRUE; }

  xdatalist = NULL;

  /* set a new tag */
  vg4->nw->xdata_retag();

reselect:
  /* select player-racket */
  if (!exec_canvas_racket(rkname, sizeof(rkname), xdatalist)) { return VG_FALSE; }

  /* show info */
  show_info("Receiving player-rackets ...");

  /* send selected player-racket to network-server */
  if (!vg4->nw->xdata_send(rkname, strlen(rkname) + 1)) { return VG_FALSE; }

  /* receive player-rackets of each client from network-server */
  if (!vg4->nw->xdata_allclients_recv(&xdatalist, exclusive, rkname, strlen(rkname) + 1, &redo)) { return VG_FALSE; }
  if (redo) {  /* select another player-racket */
    show_info("Selected racket is no more available");
    sleep(3);
    vg4->window->clear();
    goto reselect;
  }

  /* create image array and load images */
  *imgpp = calloc(clmax, sizeof(**imgpp));
  if (*imgpp == NULL) { fprintf(stderr, "calloc: %s\n", strerror(errno)); return VG_FALSE; }
  for (idx = 0; idx < clmax; idx++) { (*imgpp)[idx] = NULL; }
  for (xdptr = xdatalist; xdptr != NULL; xdptr = xdptr->next) {
    snprintf(filename, sizeof(filename), "files/player-%s.bmp", xdptr->data);
    (*imgpp)[xdptr->clnr - 1] = vg4->image->load(filename);
  }

  /* free exchange-data list */
  vg4->nw->xdata_allclients_free(xdatalist);

  return VG_TRUE;
}
