/* VgaGames4 tutorial 3 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames4.h>

static VG_BOOL show_help(void);
static VG_BOOL check_for_collision(const struct VG_Rect *, int *, const struct VG_Rect *);

/* keys */
struct {
  int k_quit_lalt;  /* part of quit: Left-ALT */
  int k_quit_q;     /* part of quit: Q */
  int k_pause;      /* pause */
  int k_up;         /* move up */
  int k_down;       /* move down */
} kref;


/* show help-text */
static VG_BOOL
show_help(void)
{
  char helptext[512];
  size_t hsize;
  struct VG_Image *imgp;
  int k_space;

  /* create text */
  snprintf(helptext, sizeof(helptext), "%%{txt&center[font=sysw:low fgcolor=0xffff00 bold=on]: Pingpong game%%}\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "\n\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [ALT+Q]%%}        Quit game\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [Cursor-UP]%%}    Move up\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [Cursor-DOWN]%%}  Move down\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [P]%%}            Pause\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "\n\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt&center[fgcolor=0xdddddd]: Press space-key%%}\n");
  hsize = strlen(helptext);

  /* create image from text */
  imgp = vg4->font->totext(helptext, "[fgcolor=0xbbbb00 boxwidth=70%%]", NULL, NULL, NULL);
  if (imgp == NULL) { return VG_FALSE; }

  /* set space-key */
  if ((k_space = vg4->input->key_insert("Spacekey", VG_FALSE, VG_FALSE)) == 0) { return VG_FALSE; }
  vg4->input->key_setkbd(k_space, VG_INPUT_KBDCODE_SPACE);

  /* copy image to window */
  vg4->window->clear();
  vg4->window->copy(imgp, NULL, NULL);
  vg4->image->destroy(imgp);

  /* wait for pressing space-key */
  for (;;) {
    if (!vg4->input->update(VG_TRUE)) { return VG_FALSE; }
    if (vg4->input->key_newpressed(k_space)) { break; }
    vg4->window->flush();
    vg4->misc->wait_time(70);
  }
  vg4->window->clear();
  vg4->window->flush();

  /* remove space-key */
  vg4->input->key_remove(k_space);

  return VG_TRUE;
}


/* check whether ball collides with player
 * @param ball_rect    rectangle of the ball
 * @param ball_angle   angle of the ball
 * @param player_rect  rectangle of the player-racket
 * @return  VG_TRUE = collision or VG_FALSE = no collision
 */
static VG_BOOL
check_for_collision(const struct VG_Rect *ball_rect, int *ball_angle, const struct VG_Rect *player_rect)
{
  VG_BOOL retw;

  if (ball_rect == NULL || ball_angle == NULL || player_rect == NULL) { return VG_FALSE; }

  /* check if ball touches player */
  if (ball_rect->x + ball_rect->w - 1 < player_rect->x || ball_rect->x > player_rect->x + player_rect->w - 1) { return VG_FALSE; }
  if (ball_rect->y + ball_rect->h - 1 < player_rect->y || ball_rect->y > player_rect->y + player_rect->h - 1) { return VG_FALSE; }

  /* modify ball-angle according to collision side */

  retw = VG_FALSE;

  if (ball_rect->x <= player_rect->x + player_rect->w - 1 && ball_rect->x + ball_rect->w - 1 >= player_rect->x + player_rect->w - 1) {
    /* ball collides with player on the right */
    *ball_angle = 360 - (*ball_angle);
    *ball_angle %= 360; *ball_angle += 360; *ball_angle %= 360;
    retw = VG_TRUE;
  }
  if (ball_rect->y + ball_rect->h - 1 >= player_rect->y && ball_rect->y <= player_rect->y) {
    /* ball collides with player on the top */
    *ball_angle = 180 - (*ball_angle);
    *ball_angle %= 360; *ball_angle += 360; *ball_angle %= 360;
    retw = VG_TRUE;
  }
  if (ball_rect->y <= player_rect->y + player_rect->h - 1 && ball_rect->y + ball_rect->h - 1 >= player_rect->y + player_rect->h - 1) {
    /* ball collides with player on the bottom */
    *ball_angle = 180 - (*ball_angle);
    *ball_angle %= 360; *ball_angle += 360; *ball_angle %= 360;
    retw = VG_TRUE;
  }

  return retw;
}


int main(int argc, char **argv) {
  const int border_size = 10;  /* size of borders in pixels */
  int winw, winh;  /* window's width and height */
  int audc_hit, audc_gameover, audc_bgmusic;  /* audio descriptors */
  struct {  /* player's structure */
    struct VG_Image *imgp;  /* image */
    struct VG_Rect rect;    /* rectangle position */
  } player;
  struct {  /* ball's structure */
    struct VG_Image *imgp;     /* image */
    struct VG_RectCent rectc;  /* rectangle position */
    int angle;                 /* moving direction angle */
    int factor;                /* moving velocity in 1/1000 pixels */
  } ball;
  struct VG_Rect rect;
  struct VG_RectCent *rectcp;
  int rectc_nr, ipos;
  struct VG_Position posi;

  (void)argc; (void)argv;

  /* initialize and open window */
  if (!VG_init("VgaGames4 tutorial 3")) { exit(1); }
  if (!vg4->window->open(VG_WINDOW_SIZE_LOW, VG_WINDOW_SCALE_BEST)) { VG_dest(); exit(1); }
  vg4->window->getsize(&winw, &winh);

  /* set mouse grabbing to off */
  vg4->input->mouse_grabbing(VG_FALSE);

  /* open audio system */
  if (!vg4->audio->open(VG_AUDIO_FREQ_MEDIUM, VG_FALSE)) { VG_dest(); exit(1); }

  /* load audio files */
  audc_hit = vg4->audio->load("files/hit.wav", 100, VG_AUDIO_VOLUME_SOUND);
  audc_gameover = vg4->audio->load("files/gameover.wav", 100, VG_AUDIO_VOLUME_SOUND);
  audc_bgmusic = vg4->audio->load("files/bgmusic.wav", 50, VG_AUDIO_VOLUME_MUSIC);

  /* set keys */
  /* quit with ALT+Q, not changeable */
  if ((kref.k_quit_lalt = vg4->input->key_insert("Quit-LALT", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_quit_lalt, VG_INPUT_KBDCODE_LALT);
  if ((kref.k_quit_q = vg4->input->key_insert("Quit-Q", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_quit_q, VG_INPUT_KBDCODE_Q);
  /* pause with P, changeable */
  if ((kref.k_pause = vg4->input->key_insert("Pause", VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_pause, VG_INPUT_KBDCODE_P);
  /* move-up with cursor-key UP, changeable */
  if ((kref.k_up = vg4->input->key_insert("Move up", VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_up, VG_INPUT_KBDCODE_UCURS);
  vg4->input->key_setgc(kref.k_up, 0, VG_INPUT_GCAXIS_RIGHTY_UP);
  /* move-down with cursor-key DOWN, changeable */
  if ((kref.k_down = vg4->input->key_insert("Move down", VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_down, VG_INPUT_KBDCODE_DCURS);
  vg4->input->key_setgc(kref.k_down, 0, VG_INPUT_GCAXIS_RIGHTY_DOWN);

  /* initialize player-racket, put it at the left */
  player.imgp = vg4->image->load("files/player.bmp");
  if (player.imgp == NULL) { VG_dest(); exit(1); }
  vg4->image->getsize(player.imgp, NULL, &player.rect.w, &player.rect.h);
  player.rect.x = 0;
  player.rect.y = (winh - player.rect.h) / 2;

  /* initialize ball, put it at the center of the window */
  ball.imgp = vg4->image->load("files/ball.bmp");
  if (ball.imgp == NULL) { VG_dest(); exit(1); }
  vg4->image->getsize(ball.imgp, NULL, &ball.rectc.rect.w, &ball.rectc.rect.h);
  ball.rectc.rect.x = (winw - ball.rectc.rect.w) / 2;
  ball.rectc.rect.y = (winh - ball.rectc.rect.h) / 2;
  ball.rectc.centx = ball.rectc.centy = 0;
  ball.angle = 45 + 90 * vg4->random->get("ball-angle", 0, 3);  /* randomized moving direction */
  ball.factor = 3500;

  /* show help */
  if (!show_help()) { VG_dest(); exit(1); }

  /* play background music looping */
  vg4->audio->play(audc_bgmusic, VG_TRUE, VG_TRUE);

  /* game loop */
  for (;;) {
    /* retrieve input-events */
    if (!vg4->input->update(VG_TRUE)) { goto endgame; }

    /* quit? */
    if (vg4->input->key_newpressed(kref.k_quit_q) && vg4->input->key_pressed(kref.k_quit_lalt)) { goto endgame; }

    /* pause? */
    if (vg4->input->key_newpressed(kref.k_pause)) {
      if (!vg4->misc->pause()) { goto endgame; }
    }


    /* +++ player: check for key-strokes +++ */

    /* moving up */
    if (vg4->input->key_pressed(kref.k_up)) {
      int y = player.rect.y;
      player.rect.y -= 2;
      if (player.rect.y < border_size) { player.rect.y = border_size; }
      /* check for collision with ball */
      if (check_for_collision(&ball.rectc.rect, &ball.angle, &player.rect)) {
        player.rect.y = y;
        vg4->audio->play(audc_hit, VG_FALSE, VG_FALSE);
      }
    }

    /* moving down */
    if (vg4->input->key_pressed(kref.k_down)) {
      int y = player.rect.y;
      player.rect.y += 2;
      if (player.rect.y > winh - border_size - player.rect.h) { player.rect.y = winh - border_size - player.rect.h; }
      /* check for collision with ball */
      if (check_for_collision(&ball.rectc.rect, &ball.angle, &player.rect)) {
        player.rect.y = y;
        vg4->audio->play(audc_hit, VG_FALSE, VG_FALSE);
      }
    }


    /* +++ move ball +++ */

    /* increase ball-factor for 1/1000 pixel */
    ball.factor++;

    /* get moving steps */
    rectc_nr = vg4->misc->moving_one_step(&ball.rectc, ball.angle, ball.factor / 100, &rectcp);

    /* iterate moving steps and check for collisions */
    for (ipos = 0; ipos < rectc_nr; ipos++) {
      /* player missed the ball? */
      if (rectcp[ipos].rect.x + ball.rectc.rect.w <= 0) {
        /* play audio and wait for its end */
        vg4->audio->stop(audc_bgmusic, VG_FALSE);
        vg4->audio->play(audc_gameover, VG_FALSE, VG_FALSE);
        while (vg4->audio->is_playing(audc_gameover, NULL)) {
          vg4->window->flush();
          vg4->misc->wait_time(40);
        }
        goto endgame;
      }

      /* check right border */
      if (rectcp[ipos].rect.x > winw - border_size - ball.rectc.rect.w) {
        ball.angle = 360 - ball.angle;
        ball.angle %= 360; ball.angle += 360; ball.angle %= 360;
        vg4->audio->play(audc_hit, VG_FALSE, VG_FALSE);
        break;
      }

      /* check top border */
      if (rectcp[ipos].rect.y < border_size) {
        ball.angle = 180 - ball.angle;
        ball.angle %= 360; ball.angle += 360; ball.angle %= 360;
        vg4->audio->play(audc_hit, VG_FALSE, VG_FALSE);
        break;
      }

      /* check bottom border */
      if (rectcp[ipos].rect.y > winh - border_size - ball.rectc.rect.h) {
        ball.angle = 180 - ball.angle;
        ball.angle %= 360; ball.angle += 360; ball.angle %= 360;
        vg4->audio->play(audc_hit, VG_FALSE, VG_FALSE);
        break;
      }

      /* check for collision with player */
      if (check_for_collision(&rectcp[ipos].rect, &ball.angle, &player.rect)) {
        vg4->audio->play(audc_hit, VG_FALSE, VG_FALSE);
        break;
      }

      /* actualize rectangle position */
      ball.rectc = rectcp[ipos];
    }


    /* +++ draw out +++ */

    /* clear window */
    vg4->window->clear();

    /* draw background and borders */
    vg4->window->fill(vg4->misc->colorbrightness(VG_COLOR_BLUE, 50));
    /* top border */
    rect.x = rect.y = 0; rect.w = winw; rect.h = border_size;
    vg4->window->draw_rect(&rect, VG_COLOR_RGB(0xb1, 0, 0), VG_TRUE);
    /* bottom border */
    rect.x = 0; rect.y = winh - border_size; rect.w = winw; rect.h = border_size;
    vg4->window->draw_rect(&rect, VG_COLOR_RGB(0xb1, 0, 0), VG_TRUE);
    /* right border */
    rect.x = winw - border_size; rect.y = 0; rect.w = border_size; rect.h = winh;
    vg4->window->draw_rect(&rect, VG_COLOR_RGB(0xb1, 0, 0), VG_TRUE);

    /* draw player */
    vg4->window->copy(player.imgp, vg4->misc->rect2position(&posi, &player.rect), NULL);

    /* draw ball */
    vg4->window->copy(ball.imgp, vg4->misc->rect2position(&posi, &ball.rectc.rect), NULL);


    /* flush contents to window and wait */
    vg4->window->flush();
    vg4->misc->wait_time(40);
  }

endgame:
  /* destroy and exit */
  VG_dest();
  exit(0);
}
