/* VgaGames4 tutorial 4 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames4.h>

static VG_BOOL show_help(void);
static VG_BOOL check_for_collisions(unsigned int, const struct VG_Rect *, int *);

/* keys */
struct {
  int k_quit_lalt;  /* part of quit: Left-ALT */
  int k_quit_q;     /* part of quit: Q */
  int k_pause;      /* pause */
  int k_up;         /* move up */
  int k_down;       /* move down */
} kref;

/* instance-IDs for each object-instance */
enum { INSTANCE_PLAYER = 1, INSTANCE_BALL, INSTANCE_BORDER_TOP, INSTANCE_BORDER_BOTTOM, INSTANCE_BORDER_RIGHT };

/* collision-tag */
unsigned int coll_tag;


/* show help-text */
static VG_BOOL
show_help(void)
{
  char helptext[512];
  size_t hsize;
  struct VG_Image *imgp;
  int k_space;

  /* create text */
  snprintf(helptext, sizeof(helptext), "%%{txt&center[font=sysw:low fgcolor=0xffff00 bold=on]: Pingpong game%%}\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "\n\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [ALT+Q]%%}        Quit game\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [Cursor-UP]%%}    Move up\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [Cursor-DOWN]%%}  Move down\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt[fgcolor=0x888888]: [P]%%}            Pause\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "\n\n");
  hsize = strlen(helptext);
  snprintf(helptext + hsize, sizeof(helptext) - hsize, "%%{txt&center[fgcolor=0xdddddd]: Press space-key%%}\n");
  hsize = strlen(helptext);

  /* create image from text */
  imgp = vg4->font->totext(helptext, "[fgcolor=0xbbbb00 boxwidth=70%%]", NULL, NULL, NULL);
  if (imgp == NULL) { return VG_FALSE; }

  /* set space-key */
  if ((k_space = vg4->input->key_insert("Spacekey", VG_FALSE, VG_FALSE)) == 0) { return VG_FALSE; }
  vg4->input->key_setkbd(k_space, VG_INPUT_KBDCODE_SPACE);

  /* copy image to window */
  vg4->window->clear();
  vg4->window->copy(imgp, NULL, NULL);
  vg4->image->destroy(imgp);

  /* wait for pressing space-key */
  for (;;) {
    if (!vg4->input->update(VG_TRUE)) { return VG_FALSE; }
    if (vg4->input->key_newpressed(k_space)) { break; }
    vg4->window->flush();
    vg4->misc->wait_time(70);
  }
  vg4->window->clear();
  vg4->window->flush();

  /* remove space-key */
  vg4->input->key_remove(k_space);

  return VG_TRUE;
}


/* check for collisions
 * @param instanceid   instance-ID of moving object
 * @param moving_rect  actual rectangle of moving object
 * @param ball_angle   angle of the ball, will be updated
 * @return  VG_TRUE = collision or VG_FALSE = no collision
 */
static VG_BOOL
check_for_collisions(unsigned int instanceid, const struct VG_Rect *moving_rect, int *ball_angle)
{ 
  struct VG_Coll *collp;
  int ipos, canz;
  VG_BOOL retw;

  if (instanceid == 0 || moving_rect == NULL || ball_angle == NULL) { return VG_FALSE; }

  retw = VG_FALSE;

  /* get collisions */
  canz = vg4->collision->setpos(coll_tag, instanceid, moving_rect, &collp);

  /* iterate collisions */
  for (ipos = 0; ipos < canz; ipos++) {
    if (collp[ipos].type == VG_COLL_TYPE_ENTRY) {  /* entering into collision */
      if (instanceid == INSTANCE_PLAYER) {
        if (collp[ipos].instanceid == INSTANCE_BALL) {  /* player collides with ball */
          /* check collision sides of ball */
          if (collp[ipos].side & VG_COLL_SIDE_LEFT) {
            *ball_angle = 360 - (*ball_angle);
            *ball_angle %= 360; *ball_angle += 360; *ball_angle %= 360;
            retw = VG_TRUE;
          }
          if (collp[ipos].side & (VG_COLL_SIDE_TOP | VG_COLL_SIDE_BOTTOM)) {
            *ball_angle = 180 - (*ball_angle);
            *ball_angle %= 360; *ball_angle += 360; *ball_angle %= 360;
            retw = VG_TRUE;
          }
        } else if (collp[ipos].instanceid == INSTANCE_BORDER_TOP
                   || collp[ipos].instanceid == INSTANCE_BORDER_BOTTOM) {  /* player collides with border */
          retw = VG_TRUE;
        }
      } else if (instanceid == INSTANCE_BALL) {
        if (collp[ipos].instanceid == INSTANCE_PLAYER) {  /* ball collides with player */
          /* check collision sides of player */
          if (collp[ipos].side & VG_COLL_SIDE_RIGHT) {
            *ball_angle = 360 - (*ball_angle);
            *ball_angle %= 360; *ball_angle += 360; *ball_angle %= 360;
            retw = VG_TRUE;
          }
          if (collp[ipos].side & (VG_COLL_SIDE_TOP | VG_COLL_SIDE_BOTTOM)) {
            *ball_angle = 180 - (*ball_angle);
            *ball_angle %= 360; *ball_angle += 360; *ball_angle %= 360;
            retw = VG_TRUE;
          }
        } else if (collp[ipos].instanceid == INSTANCE_BORDER_TOP
                   || collp[ipos].instanceid == INSTANCE_BORDER_BOTTOM) {  /* ball collides with top- or bottom-border */
          *ball_angle = 180 - (*ball_angle);
          *ball_angle %= 360; *ball_angle += 360; *ball_angle %= 360;
          retw = VG_TRUE;
        } else if (collp[ipos].instanceid == INSTANCE_BORDER_RIGHT) {  /* ball collides with right border */
          *ball_angle = 360 - (*ball_angle);
          *ball_angle %= 360; *ball_angle += 360; *ball_angle %= 360;
          retw = VG_TRUE;
        }
      }
    }
  }

  /* free collision struct */
  if (collp != NULL) { free(collp); }

  return retw;
}


int main(int argc, char **argv) {
  const int border_size = 10;  /* size of borders in pixels */
  int winw, winh;  /* window's width and height */
  int audc_hit, audc_gameover, audc_bgmusic;  /* audio descriptors */
  struct {  /* player's structure */
    struct VG_Image *imgp;  /* image */
    struct VG_Rect rect;    /* rectangle position */
  } player;
  struct {  /* ball's structure */
    struct VG_Image *imgp;     /* image */
    struct VG_RectCent rectc;  /* rectangle position */
    int angle;                 /* moving direction angle */
    int factor;                /* moving velocity in 1/1000 pixels */
  } ball;
  struct {  /* border's structure */
    struct VG_Image *imgp;  /* image */
    struct VG_Rect rect;    /* rectangle position */
  } border[3];
  struct VG_RectCent *rectcp;
  int rectc_nr, ipos;
  struct VG_Position posi;

  (void)argc; (void)argv;

  /* initialize and open window */
  if (!VG_init("VgaGames4 tutorial 4")) { exit(1); }
  if (!vg4->window->open(VG_WINDOW_SIZE_LOW, VG_WINDOW_SCALE_BEST)) { VG_dest(); exit(1); }
  vg4->window->getsize(&winw, &winh);

  /* set mouse grabbing to off */
  vg4->input->mouse_grabbing(VG_FALSE);

  /* open audio system */
  if (!vg4->audio->open(VG_AUDIO_FREQ_MEDIUM, VG_FALSE)) { VG_dest(); exit(1); }

  /* load audio files */
  audc_hit = vg4->audio->load("files/hit.wav", 100, VG_AUDIO_VOLUME_SOUND);
  audc_gameover = vg4->audio->load("files/gameover.wav", 100, VG_AUDIO_VOLUME_SOUND);
  audc_bgmusic = vg4->audio->load("files/bgmusic.wav", 50, VG_AUDIO_VOLUME_MUSIC);

  /* set keys */
  /* quit with ALT+Q, not changeable */
  if ((kref.k_quit_lalt = vg4->input->key_insert("Quit-LALT", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_quit_lalt, VG_INPUT_KBDCODE_LALT);
  if ((kref.k_quit_q = vg4->input->key_insert("Quit-Q", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_quit_q, VG_INPUT_KBDCODE_Q);
  /* pause with P, changeable */
  if ((kref.k_pause = vg4->input->key_insert("Pause", VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_pause, VG_INPUT_KBDCODE_P);
  /* move-up with cursor-key UP, changeable */
  if ((kref.k_up = vg4->input->key_insert("Move up", VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_up, VG_INPUT_KBDCODE_UCURS);
  vg4->input->key_setgc(kref.k_up, 0, VG_INPUT_GCAXIS_RIGHTY_UP);
  /* move-down with cursor-key DOWN, changeable */
  if ((kref.k_down = vg4->input->key_insert("Move down", VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(kref.k_down, VG_INPUT_KBDCODE_DCURS);
  vg4->input->key_setgc(kref.k_down, 0, VG_INPUT_GCAXIS_RIGHTY_DOWN);

  /* create collision-tag */
  coll_tag = vg4->collision->create(NULL, 0, 0);

  /* initialize borders */
  /* top border */
  border[0].rect.x = border[0].rect.y = 0; border[0].rect.w = winw; border[0].rect.h = border_size;
  border[0].imgp = vg4->image->create(border[0].rect.w, border[0].rect.h);
  vg4->image->fill(border[0].imgp, VG_COLOR_RGB(0xb1, 0, 0));
  vg4->collision->insert(coll_tag, INSTANCE_BORDER_TOP, &border[0].rect, 100);  /* insert into collision-tag */
  /* bottom border */
  border[1].rect.x = 0; border[1].rect.y = winh - border_size; border[1].rect.w = winw; border[1].rect.h = border_size;
  border[1].imgp = vg4->image->create(border[1].rect.w, border[1].rect.h);
  vg4->image->fill(border[1].imgp, VG_COLOR_RGB(0xb1, 0, 0));
  vg4->collision->insert(coll_tag, INSTANCE_BORDER_BOTTOM, &border[1].rect, 100);  /* insert into collision-tag */
  /* right border */
  border[2].rect.x = winw - border_size; border[2].rect.y = 0; border[2].rect.w = border_size; border[2].rect.h = winh;
  border[2].imgp = vg4->image->create(border[2].rect.w, border[2].rect.h);
  vg4->image->fill(border[2].imgp, VG_COLOR_RGB(0xb1, 0, 0));
  vg4->collision->insert(coll_tag, INSTANCE_BORDER_RIGHT, &border[2].rect, 100);  /* insert into collision-tag */

  /* initialize player-racket, put it at the left */
  player.imgp = vg4->image->load("files/player.bmp");
  if (player.imgp == NULL) { VG_dest(); exit(1); }
  vg4->image->getsize(player.imgp, NULL, &player.rect.w, &player.rect.h);
  player.rect.x = 0;
  player.rect.y = (winh - player.rect.h) / 2;
  /* insert player-racket with 90% of size into collision-tag */
  vg4->collision->insert(coll_tag, INSTANCE_PLAYER, &player.rect, 90);

  /* initialize ball, put it at the center of the window */
  ball.imgp = vg4->image->load("files/ball.bmp");
  if (ball.imgp == NULL) { VG_dest(); exit(1); }
  vg4->image->getsize(ball.imgp, NULL, &ball.rectc.rect.w, &ball.rectc.rect.h);
  ball.rectc.rect.x = (winw - ball.rectc.rect.w) / 2;
  ball.rectc.rect.y = (winh - ball.rectc.rect.h) / 2;
  ball.rectc.centx = ball.rectc.centy = 0;
  ball.angle = 45 + 90 * vg4->random->get("ball-angle", 0, 3);  /* randomized moving direction */
  ball.factor = 3500;
  /* insert ball with 90% of size into collision-tag */
  vg4->collision->insert(coll_tag, INSTANCE_BALL, &ball.rectc.rect, 90);

  /* show help */
  if (!show_help()) { VG_dest(); exit(1); }

  /* play background music looping */
  vg4->audio->play(audc_bgmusic, VG_TRUE, VG_TRUE);

  /* game loop */
  for (;;) {
    /* retrieve input-events */
    if (!vg4->input->update(VG_TRUE)) { goto endgame; }

    /* quit? */
    if (vg4->input->key_newpressed(kref.k_quit_q) && vg4->input->key_pressed(kref.k_quit_lalt)) { goto endgame; }

    /* pause? */
    if (vg4->input->key_newpressed(kref.k_pause)) {
      if (!vg4->misc->pause()) { goto endgame; }
    }


    /* +++ player: check for key-strokes +++ */

    /* moving up */
    if (vg4->input->key_pressed(kref.k_up)) {
      int y = player.rect.y;
      player.rect.y -= 2;
      /* check for collisions */
      if (check_for_collisions(INSTANCE_PLAYER, &player.rect, &ball.angle)) {
        /* set back to previous position */
        player.rect.y = y;
        vg4->collision->setpos(coll_tag, INSTANCE_PLAYER, &player.rect, NULL);
        vg4->audio->play(audc_hit, VG_FALSE, VG_FALSE);
      }
    }

    /* moving down */
    if (vg4->input->key_pressed(kref.k_down)) {
      int y = player.rect.y;
      player.rect.y += 2;
      /* check for collisions */
      if (check_for_collisions(INSTANCE_PLAYER, &player.rect, &ball.angle)) {
        /* set back to previous position */
        player.rect.y = y;
        vg4->collision->setpos(coll_tag, INSTANCE_PLAYER, &player.rect, NULL);
        vg4->audio->play(audc_hit, VG_FALSE, VG_FALSE);
      }
    }


    /* +++ move ball +++ */

    /* increase ball-factor for 1/1000 pixel */
    ball.factor++;

    /* get moving steps */
    rectc_nr = vg4->misc->moving_one_step(&ball.rectc, ball.angle, ball.factor / 100, &rectcp);

    /* iterate moving steps and check for collisions */
    for (ipos = 0; ipos < rectc_nr; ipos++) {
      /* player missed the ball? */
      if (rectcp[ipos].rect.x + ball.rectc.rect.w <= 0) {
        /* play audio and wait for its end */
        vg4->audio->stop(audc_bgmusic, VG_FALSE);
        vg4->audio->play(audc_gameover, VG_FALSE, VG_FALSE);
        while (vg4->audio->is_playing(audc_gameover, NULL)) {
          vg4->window->flush();
          vg4->misc->wait_time(40);
        }
        goto endgame;
      }

      /* check for collisions */
      if (check_for_collisions(INSTANCE_BALL, &rectcp[ipos].rect, &ball.angle)) {
        /* set back to previous position */
        vg4->collision->setpos(coll_tag, INSTANCE_BALL, &ball.rectc.rect, NULL);
        vg4->audio->play(audc_hit, VG_FALSE, VG_FALSE);
        break;
      }

      /* actualize rectangle position */
      ball.rectc = rectcp[ipos];
    }


    /* +++ draw out +++ */

    /* clear window */
    vg4->window->clear();

    /* draw background and borders */
    vg4->window->fill(vg4->misc->colorbrightness(VG_COLOR_BLUE, 50));
    vg4->window->copy(border[0].imgp, vg4->misc->rect2position(&posi, &border[0].rect), NULL);
    vg4->window->copy(border[1].imgp, vg4->misc->rect2position(&posi, &border[1].rect), NULL);
    vg4->window->copy(border[2].imgp, vg4->misc->rect2position(&posi, &border[2].rect), NULL);

    /* draw player */
    vg4->window->copy(player.imgp, vg4->misc->rect2position(&posi, &player.rect), NULL);

    /* draw ball */
    vg4->window->copy(ball.imgp, vg4->misc->rect2position(&posi, &ball.rectc.rect), NULL);


    /* flush contents to window and wait */
    vg4->window->flush();
    vg4->misc->wait_time(40);
  }

endgame:
  /* destroy and exit */
  VG_dest();
  exit(0);
}
